import { WebSocketServer } from 'ws';
import jwt from 'jsonwebtoken';
import { pool } from './config/db.js';

// Mapa para rastrear conexiones de usuarios: { userId: WebSocket }
export const userConnections = new Map();

// Usaremos un Map para guardar los clientes conectados y sus datos internamente
const clients = new Map();

/**
 * Envía un mensaje a todos los clientes conectados, excepto al remitente (opcional).
 * @param {Object} data - El objeto JSON a enviar.
 * @param {string|null} excludeUserId - ID del usuario a excluir (opcional).
 */
const broadcast = (data, excludeUserId = null) => {
  const message = JSON.stringify(data);
  clients.forEach((client, id) => {
    if (id !== excludeUserId && client.ws.readyState === client.ws.OPEN) {
      client.ws.send(message);
    }
  });
};

/**
 * Configura e inicia el servidor WebSocket adjunto al servidor HTTP.
 * @param {import('http').Server} server - Instancia del servidor HTTP.
 */
export const setupWebSocketServer = (server) => {
  // +++ MEJORA: Usar noServer: true para manejar el upgrade manualmente.
  // Esto evita conflictos y errores 400 si la URL tiene ligeras variaciones.
  const wss = new WebSocketServer({ noServer: true });

  server.on('upgrade', (request, socket, head) => {
    // Parsear la URL para obtener el pathname limpio
    const { pathname } = new URL(request.url, `http://${request.headers.host}`);

    if (pathname === '/api/ws') {
      wss.handleUpgrade(request, socket, head, (ws) => {
        wss.emit('connection', ws, request);
      });
    } else {
      socket.destroy();
    }
  });

  // Función para marcar la conexión como viva cuando recibimos un 'pong'
  function heartbeat() {
    this.isAlive = true;
  }

  // Intervalo GLOBAL para verificar conexiones muertas (Ping/Pong)
  // Esto mantiene la conexión estable a través de proxies como Ngrok/Nginx
  const interval = setInterval(() => {
    wss.clients.forEach((ws) => {
      if (ws.isAlive === false) return ws.terminate();

      ws.isAlive = false;
      ws.ping();
    });
  }, 30000); // Cada 30 segundos

  wss.on('close', () => {
    clearInterval(interval);
  });

  wss.on('connection', (ws) => {
    console.log('🔌 Cliente WebSocket conectado');

    ws.isAlive = true;
    ws.on('pong', heartbeat);

    // Manejo de errores de socket para evitar caídas
    ws.on('error', (err) => console.error('⚠️ Error en conexión WS:', err));

    ws.on('message', async (message) => {
      try {
        let data;
        try {
          data = JSON.parse(message);
        } catch (e) {
          console.error('Mensaje no es JSON válido:', message);
          return;
        }

        // 1. Autenticación del cliente
        if (data.type === 'auth' && data.token) {
          await handleAuth(ws, data.token);
          return;
        }

        // Si no está autenticado, ignorar otros mensajes
        if (!ws.userId) return;

        // Despachador de eventos (Switch Case para orden)
        switch (data.type) {
          case 'location_update':
            await handleLocationUpdate(ws, data.payload);
            break;
          case 'chat_message':
            handleChatMessage(ws, data.payload);
            break;
          case 'emergency_alert':
            handleEmergencyAlert(ws);
            break;
          case 'typing_status':
            handleTypingStatus(ws, data.payload);
            break;
          case 'admin_command':
            handleAdminCommand(ws, data.payload);
            break;
          case 'status_change':
            handleStatusChange(ws, data.payload);
            break;
          case 'device_health':
            handleDeviceHealth(ws, data.payload);
            break;
          case 'get_chat_history':
            await handleGetChatHistory(ws);
            break;
          default:
            // console.warn(`Tipo de mensaje desconocido: ${data.type}`);
            break;
        }
      } catch (e) {
        console.error('Error procesando mensaje WS:', e);
      }
    });

    ws.on('close', () => {
      handleDisconnect(ws);
    });
  });

  return wss;
};

// --- Handlers Modulares (Funciones bien hechas) ---

async function handleAuth(ws, token) {
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const userId = decoded.id;

    // +++ FIX: Seleccionar * para evitar error si la columna 'rol' no existe o se llama diferente
    const [userRows] = await pool.execute('SELECT * FROM usuarios WHERE id = ?', [userId]);
    if (userRows.length === 0) {
      ws.close(1008, 'Usuario no válido');
      return;
    }
    const userName = userRows[0].usuario;
    // +++ FIX: Fallback para el rol si la columna 'rol' no existe (ej. id_rol, role, tipo)
    const userRole = userRows[0].rol_id || userRows[0].rol || userRows[0].id_rol || userRows[0].role || userRows[0].tipo_usuario || 'user';

    // Configurar propiedades del socket
    ws.userId = userId;
    ws.userName = userName;
    ws.userRole = userRole;

    // Guardar en mapas
    clients.set(userId, {
      ws,
      userName,
      userRole,
      lastLocation: null,
      operationalStatus: 'available',
      batteryLevel: null
    });
    userConnections.set(userId, ws);

    console.log(`✅ Usuario ${userName} (ID: ${userId}) autenticado en WebSocket.`);

    // 1. Enviar estado actual de OTROS usuarios al recién conectado
    clients.forEach((client, id) => {
      if (id !== userId && client.ws.readyState === client.ws.OPEN) {
        // Usuario conectado
        ws.send(JSON.stringify({
          type: 'user_connected',
          payload: {
            userId: id,
            userName: client.userName,
            connectedAt: new Date().toISOString(),
            isInitial: true
          }
        }));

        // Ubicación
        if (client.lastLocation) {
          ws.send(JSON.stringify({
            type: 'location_update',
            payload: { userId: id, userName: client.userName, ...client.lastLocation }
          }));
        }

        // Estado
        if (client.operationalStatus && client.operationalStatus !== 'available') {
          ws.send(JSON.stringify({
            type: 'status_change',
            payload: { userId: id, status: client.operationalStatus }
          }));
        }

        // Batería
        if (client.batteryLevel !== null) {
          ws.send(JSON.stringify({
            type: 'device_health',
            payload: { userId: id, batteryLevel: client.batteryLevel }
          }));
        }
      }
    });

    // 2. Notificar a TODOS que este usuario se conectó
    broadcast({
      type: 'user_connected',
      payload: {
        userId,
        userName,
        connectedAt: new Date().toISOString()
      }
    }, userId);

  } catch (err) {
    console.error('Error de autenticación WS:', err.message);
    ws.close(1008, 'Token inválido');
  }
}

async function handleLocationUpdate(ws, payload) {
  // CORRECCIÓN: Soporte para lat/lng (estándar) o latitud/longitud (posible envío desde móvil)
  const lat = payload.lat || payload.latitud;
  const lng = payload.lng || payload.longitud;
  const timestamp = payload.timestamp;

  // Actualizar memoria
  const client = clients.get(ws.userId);
  if (client && lat && lng) {
    client.lastLocation = payload;
  }

  // Persistir en DB
  if (lat && lng) {
    try {
      await pool.execute(
        'INSERT INTO historial_ubicaciones (usuario_id, ubicacion, timestamp) VALUES (?, POINT(?, ?), ?)',
        [ws.userId, lng, lat, timestamp ? new Date(timestamp) : new Date()]
      );
    } catch (dbError) {
      console.error('❌ Error al guardar ubicación en la DB:', dbError.message);
    }
  }

  // Broadcast
  broadcast({
    type: 'location_update',
    payload: {
      userId: ws.userId,
      userName: ws.userName,
      ...payload,
    }
  }, ws.userId);
}

async function handleGetChatHistory(ws) {
  try {
    // Obtener últimos 50 mensajes ordenados por fecha
    const [rows] = await pool.execute(`
      SELECT m.mensaje, m.timestamp, m.remitente_id, u.usuario as remitente_nombre
      FROM mensajes_chat m
      LEFT JOIN usuarios u ON m.remitente_id = u.id
      ORDER BY m.timestamp DESC
      LIMIT 50
    `);

    // Invertir para enviar en orden cronológico (antiguo -> nuevo)
    const history = rows.reverse().map(row => ({
      fromUserId: row.remitente_id,
      fromUserName: row.remitente_nombre || 'Desconocido',
      message: row.mensaje,
      timestamp: row.timestamp,
      isMe: row.remitente_id === ws.userId
    }));

    ws.send(JSON.stringify({ type: 'chat_history', payload: { history } }));
  } catch (err) {
    console.error('❌ Error obteniendo historial chat:', err.message);
  }
}

async function handleChatMessage(ws, payload) {
  const { message, toUserId } = payload;

  // +++ MEJORA: Guardar mensaje en Base de Datos
  try {
    await pool.execute(
      'INSERT INTO mensajes_chat (remitente_id, destinatario_id, mensaje, timestamp) VALUES (?, ?, ?, NOW())',
      [ws.userId, toUserId || null, message]
    );
  } catch (err) {
    console.error('❌ Error guardando chat en DB:', err.message);
  }

  const chatPayload = {
    type: 'chat_message',
    payload: {
      fromUserId: ws.userId,
      fromUserName: ws.userName,
      type: 'chat_message', // Identificador para el frontend
      message: message,
      timestamp: new Date().toISOString()
    }
  };

  if (toUserId) {
    const targetClient = clients.get(toUserId);
    if (targetClient && targetClient.ws.readyState === targetClient.ws.OPEN) {
      targetClient.ws.send(JSON.stringify(chatPayload));
    }
  } else {
    broadcast(chatPayload, ws.userId);
  }
}

async function handleEmergencyAlert(ws) {
  const location = clients.get(ws.userId)?.lastLocation;

  // +++ MEJORA: Registrar alerta en Base de Datos
  try {
    await pool.execute(
      'INSERT INTO alertas_emergencia (usuario_id, ubicacion, timestamp) VALUES (?, POINT(?, ?), NOW())',
      [ws.userId, location?.lng || 0, location?.lat || 0]
    );
  } catch (err) {
    console.error('❌ Error guardando alerta en DB:', err.message);
  }

  const alertMsg = {
    type: 'emergency_alert',
    payload: {
      userId: ws.userId,
      userName: ws.userName,
      status: 'emergency', // Para que el frontend active la alerta roja
      type: 'emergency_alert',
      location: location || null,
      timestamp: new Date().toISOString()
    }
  };
  // Enviar a TODOS
  clients.forEach((client) => {
    if (client.ws.readyState === client.ws.OPEN) {
      client.ws.send(JSON.stringify(alertMsg));
    }
  });
}

function handleTypingStatus(ws, payload) {
  const { isTyping, toUserId } = payload;
  const typingPayload = {
    type: 'typing_status',
    payload: {
      userId: ws.userId,
      userName: ws.userName,
      isTyping: isTyping
    }
  };

  if (toUserId) {
    const targetClient = clients.get(toUserId);
    if (targetClient && targetClient.ws.readyState === targetClient.ws.OPEN) {
      targetClient.ws.send(JSON.stringify(typingPayload));
    }
  } else {
    broadcast(typingPayload, ws.userId);
  }
}

function handleAdminCommand(ws, payload) {
  if (ws.userRole !== 'admin') {
    console.warn(`⚠️ Usuario ${ws.userName} intentó enviar comando admin sin permiso.`);
    return;
  }

  const { command, targetUserId } = payload;

  if (command === 'kick_user' && targetUserId) {
    const target = clients.get(targetUserId);
    if (target) {
      target.ws.close(4000, 'Expulsado por el administrador');
      ws.send(JSON.stringify({ type: 'admin_response', payload: { success: true, message: `Usuario ${target.userName} expulsado.` } }));
    }
  }

  if (command === 'force_location_update') {
    const cmdMsg = JSON.stringify({
      type: 'force_location_update',
      payload: { requestedBy: ws.userName }
    });

    if (targetUserId) {
      const target = clients.get(targetUserId);
      if (target) target.ws.send(cmdMsg);
    } else {
      clients.forEach((client) => {
        if (client.ws.readyState === client.ws.OPEN) client.ws.send(cmdMsg);
      });
    }
  }
}

function handleStatusChange(ws, payload) {
  const { status } = payload;
  const client = clients.get(ws.userId);
  if (client) client.operationalStatus = status;

  broadcast({
    type: 'status_change',
    payload: {
      userId: ws.userId,
      userName: ws.userName,
      status: status,
      timestamp: new Date().toISOString()
    }
  }, ws.userId);
}

function handleDeviceHealth(ws, payload) {
  const { batteryLevel } = payload;
  const client = clients.get(ws.userId);
  if (client) client.batteryLevel = batteryLevel;

  broadcast({
    type: 'device_health',
    payload: {
      userId: ws.userId,
      batteryLevel: batteryLevel
    }
  }, ws.userId);
}

function handleDisconnect(ws) {
  if (ws.userId) {
    console.log(`🔌 Usuario ${ws.userName} (ID: ${ws.userId}) desconectado.`);
    userConnections.delete(ws.userId);
    clients.delete(ws.userId);

    broadcast({
      type: 'user_disconnected',
      payload: {
        userId: ws.userId,
        userName: ws.userName,
        disconnectedAt: new Date().toISOString()
      }
    });
  } else {
    console.log('🔌 Cliente WebSocket desconectado (no autenticado).');
  }
}
