import { Parser } from 'json2csv';
import ExcelJS from 'exceljs';

/**
 * Genera y envía un archivo (CSV o XLSX) para su descarga.
 * @param {object} res - El objeto de respuesta de Express.
 * @param {Array<object>} data - Los datos a exportar.
 * @param {'csv' | 'xlsx'} format - El formato del archivo.
 * @param {string} resourceName - El nombre base para el archivo (ej. 'historial').
 * @param {Array<object>} columns - La configuración de las columnas para el archivo.
 */
export const sendExportFile = async (res, data, format, resourceName, columns) => {
  const dateStr = new Date().toISOString().slice(0, 10);
  const fileName = `${resourceName}_${dateStr}.${format}`;

  if (format === 'csv') {
    const fields = columns.map(col => col.key);
    const opts = { fields };
    try {
      const parser = new Parser(opts);
      const csv = parser.parse(data);

      res.header('Content-Type', 'text/csv');
      res.attachment(fileName);
      return res.send(csv);
    } catch (err) {
      console.error(`Error al generar el CSV para ${resourceName}:`, err);
      throw new Error('No se pudo generar el archivo CSV.');
    }
  }

  if (format === 'xlsx') {
    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet(resourceName.charAt(0).toUpperCase() + resourceName.slice(1));

    worksheet.columns = columns;
    worksheet.addRows(data);

    res.setHeader(
      'Content-Type',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    );
    res.setHeader(
      'Content-Disposition',
      `attachment; filename=${fileName}`
    );

    await workbook.xlsx.write(res);
    return res.end();
  }

  throw new Error('Formato de exportación no soportado.');
};