// Importa las librerías para procesamiento de audio y sus rutas.
import ffmpeg from 'fluent-ffmpeg';
import ffmpegPath from '@ffmpeg-installer/ffmpeg';
// Importa módulos de Node.js para manejar rutas y el sistema de archivos.
import path from 'path';
import fs from 'fs';
// Importa `sharp` para un procesamiento de imágenes rápido y eficiente.
import sharp from 'sharp';
// Importa `uuid` para generar nombres de archivo únicos.
import { v4 as uuidv4 } from 'uuid';
import { fileURLToPath } from 'url';
import { dirname } from 'path';

// --- CONFIGURACIÓN INICIAL ---

// Indica a `fluent-ffmpeg` dónde encontrar el ejecutable de ffmpeg.
ffmpeg.setFfmpegPath(ffmpegPath.path);

// Obtiene la ruta del directorio actual en un entorno de Módulos ES.
const __filename = fileURLToPath(import.meta.url);
const __dirname  = dirname(dirname(__filename)); // Sube un nivel para estar en la carpeta /src

// --- SERVICIO DE COMPRESIÓN DE IMÁGENES ---
export async function compressAndSaveImage(fileBuffer) {
  // Si no se recibe un buffer, no hay nada que hacer.
  if (!fileBuffer) return null;

  // Crea un subdirectorio con la fecha actual (YYYY-MM-DD) para organizar los archivos.
  const today = new Date().toISOString().slice(0, 10);
  const dir = path.join(__dirname, '..', 'uploads', today); // Navega fuera de /src a la carpeta /uploads
  // Si el directorio no existe, lo crea de forma recursiva.
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });

  // Genera un nombre de archivo único para evitar colisiones.
  const filename = `${Date.now()}_${uuidv4()}.jpeg`;
  const filepath = path.join(dir, filename);

  // Usa `sharp` para procesar la imagen desde el buffer.
  await sharp(fileBuffer)
    // Redimensiona la imagen para que no exceda 1400x1400, manteniendo la proporción.
    .resize({ width: 1400, height: 1400, fit: 'inside', withoutEnlargement: true })
    // Convierte la imagen a formato JPEG con una calidad del 80% y optimizaciones.
    .jpeg({ quality: 80, progressive: true, optimizeScans: true })
    // Guarda el resultado en el disco.
    .toFile(filepath);

  // Devuelve la ruta pública del archivo para guardarla en la base de datos.
  return `/uploads/${today}/${filename}`;
}

// --- SERVICIO DE COMPRESIÓN DE AUDIO ---
export async function compressAndSaveAudio(fileBuffer) {
  if (!fileBuffer) return null;

  const today = new Date().toISOString().slice(0, 10);
  const dir = path.join(__dirname, '..', 'uploads', today);
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });

  // Ffmpeg trabaja con archivos, no con buffers directamente. Por lo tanto, se crea un archivo temporal.
  const tempInputPath = path.join(dir, `temp_audio_${uuidv4()}`);
  const outputPath = path.join(dir, `${Date.now()}_${uuidv4()}.mp3`);

  // Escribe el buffer del audio en el archivo temporal.
  fs.writeFileSync(tempInputPath, fileBuffer);

  // Devuelve una Promesa, ya que ffmpeg funciona con eventos.
  return new Promise((resolve, reject) => {
    ffmpeg(tempInputPath)
      .audioCodec('libmp3lame') // Usa el codificador MP3 LAME.
      .audioBitrate('96k')      // Comprime el audio a un bitrate de 96 kbps.
      .toFormat('mp3')           // Define el formato de salida.
      .on('error', (err) => {
        fs.unlinkSync(tempInputPath); // Asegúrate de borrar el archivo temporal si hay un error.
        reject(new Error('Error al procesar el audio: ' + err.message));
      })
      .on('end', () => {
        fs.unlinkSync(tempInputPath); // Borra el archivo temporal después de una conversión exitosa.
        resolve(`/uploads/${today}/${path.basename(outputPath)}`);
      })
      .save(outputPath);
  });
}