// server.js (ESM)
// ----------------------------
// 1) Variables de entorno
import 'dotenv/config';

// +++ Importaciones para WebSocket
import http from 'http';
import https from 'https';
import fs from 'fs';
import { pool } from './config/db.js';
// +++ Importar la función de configuración de WebSockets
import { setupWebSocketServer } from './websockets.js';
console.log(`[ENV] Iniciando en modo: ${process.env.NODE_ENV || 'development'}`);

// 2) App y DB
import app from './app.js';

// 3) Ruta /health (si tu app.js no la define)
if (typeof app?.get === 'function') {
  app.get('/health', (req, res) => {
    res.status(200).json({
      ok: true,
      uptime: process.uptime(),
      timestamp: new Date().toISOString(),
    });
  });
}

// --- MEJORA: Función para verificar la conexión a la DB con reintentos ---
const checkDbConnection = async (retries = 5, delay = 2000) => {
  while (retries > 0) {
    try {
      const conn = await pool.getConnection();
      await conn.ping();
      conn.release();
      console.log('[DB] Conectado a MySQL - OK');
      return; // Conexión exitosa, salimos de la función
    } catch (err) {
      console.error(`[DB] Error de conexión - FALLO. Reintentos restantes: ${retries - 1}`, err?.message || err);
      retries--;
      if (retries === 0) {
        // Si se agotan los reintentos, lanzamos el error para detener la app.
        throw new Error("No se pudo conectar a la base de datos después de varios intentos.");
      }
      // Espera exponencial antes de reintentar (2s, 4s, 8s, ...)
      await new Promise(res => setTimeout(res, delay));
      delay *= 2;
    }
  }
};

// 4) Verificación temprana de DB con reintentos
(async () => {
  try {
    await checkDbConnection();
  } catch (err) {
    console.error(err.message);
    process.exit(1);
  }
})();

// 5) Lanzar servidor
// Puerto 3000 como predeterminado (fallback)
const PORT = process.env.PORT || 3000;
// *** CLAVE: Usar '0.0.0.0' para escuchar peticiones externas.
const HOST = '0.0.0.0';

// +++ Crear servidor HTTP explícito para poder adjuntar WebSockets
let server;

if (process.env.SSL_KEY_PATH && process.env.SSL_CERT_PATH) {
  try {
    const options = {
      key: fs.readFileSync(process.env.SSL_KEY_PATH),
      cert: fs.readFileSync(process.env.SSL_CERT_PATH),
    };
    server = https.createServer(options, app);
    console.log('🔒 Servidor HTTPS habilitado');
  } catch (e) {
    console.error('⚠️ Error cargando certificados SSL, usando HTTP:', e.message);
    server = http.createServer(app);
  }
} else {
  server = http.createServer(app);
}

// +++ INICIO: Lógica del Servidor WebSocket (Desestructurada)
setupWebSocketServer(server);
// +++ FIN: Lógica del Servidor WebSocket

server.listen(PORT, HOST, () => {
  const time = new Date().toLocaleString();
  const protocol = server instanceof https.Server ? 'https' : 'http';
  console.log(`[${time}] MediApp escuchando en ${protocol}://${HOST}:${PORT}`);
});

// 6) Manejo de errores no atrapados / apagado limpio
process.on('unhandledRejection', (reason) => {
  console.error('ADVERTENCIA: UnhandledRejection:', reason);
});
process.on('uncaughtException', (err) => {
  console.error('ADVERTENCIA: UncaughtException:', err);
});
const shutdown = (signal) => () => {
  console.log(`[APAGADO] Recibido ${signal}, cerrando servidor...`);
  server.close(() => { // server.close() cerrará tanto HTTP como WS
    console.log('Servidor cerrado. Bye!');
    process.exit(0);
  });
};
process.on('SIGINT', shutdown('SIGINT'));
process.on('SIGTERM', shutdown('SIGTERM'));
