import Joi from 'joi';

// Lista blanca de tipos de registro válidos para el filtro.
const tiposDeRegistroValidos = [
  'corte_servicio',
  'entregas_individuales',
  'entregas_edificio',
  'infraestructura',
  'medidores',
  'notificadores'
];

export const historialGlobalSchema = Joi.object({
  // Filtros opcionales
  nombre_usuario: Joi.string().optional().allow(''),
  usuario_id: Joi.number().integer().min(1).optional().allow(''),
  tipo_registro: Joi.string().valid(...tiposDeRegistroValidos).optional().allow(''),
  fecha_inicio: Joi.date().iso().optional().allow(''),
  // `fecha_fin` debe ser igual o posterior a `fecha_inicio` si ambas existen.
  fecha_fin: Joi.date().iso().when('fecha_inicio', {
    // Solo aplicar la regla 'min' si 'fecha_inicio' es una fecha válida.
    is: Joi.date().iso().required(),
    then: Joi.date().min(Joi.ref('fecha_inicio')).optional().allow(''),
    otherwise: Joi.optional().allow('')
  }),
  // Valida el formato de exportación
  formato: Joi.string().valid('csv', 'xlsx').optional(),

  // Paginación
  page: Joi.number().integer().min(1).optional(),
  limit: Joi.number().integer().min(1).optional()
});

export const usuarioIdSchema = Joi.object({
  id: Joi.number().integer().min(1).required()
});

export const updateEstadoUsuarioSchema = Joi.object({
  // estatus_id debe ser 0 (inactivo) o 1 (activo).
  estatus_id: Joi.number().valid(0, 1).required()
});

export const updateUsuarioSchema = Joi.object({
  nombre: Joi.string().min(2).max(50).required(),
  apellido_paterno: Joi.string().min(2).max(50).required(),
  apellido_materno: Joi.string().min(2).max(50).required(),
  numero_empleado: Joi.string().pattern(/^[0-9]+$/).required().messages({
    'string.pattern.base': `"numero_empleado" solo debe contener números`
  }),
  usuario: Joi.string().required(),
  rol_id: Joi.number().valid(1, 2, 3).required()
});

export const crearUsuarioSchema = Joi.object({
  nombre: Joi.string().min(2).max(50).required(),
  apellido_paterno: Joi.string().min(2).max(50).required(),
  apellido_materno: Joi.string().min(2).max(50).required(),
  numero_empleado: Joi.string().pattern(/^[0-9]+$/).required().messages({
    'string.pattern.base': `"numero_empleado" solo debe contener números`
  }),
  usuario: Joi.string().required(),
  password: Joi.string().min(6).required(),
  rol_id: Joi.number().valid(1, 2, 3).required(), // 1: Admin, 2: Jefe de Area, 3: Empleado
  estatus_id: Joi.number().valid(0, 1).default(1) // Por defecto, activo
});

export const deleteRegistroSchema = Joi.object({
  tabla: Joi.string().valid(...tiposDeRegistroValidos).required(),
  id: Joi.number().integer().min(1).required()
});

export const updateConfiguracionesSchema = Joi.object({
  // Define las claves de configuración permitidas y sus tipos.
  tiempo_sincronizacion: Joi.number().integer().min(10).optional(),
  calidad_imagen: Joi.number().integer().min(10).max(100).optional(),
  gps_habilitado: Joi.boolean().optional(),
  max_intentos_sincronizacion: Joi.number().integer().min(1).max(10).optional(),
  // Añadimos la nueva clave de configuración y su validación
  app_version_requerida: Joi.string().pattern(/^\d+\.\d+\.\d+$/).optional().messages({
    'string.pattern.base': 'El formato de la versión debe ser X.Y.Z (ej: 1.0.0)'
  })
})
  // Asegura que el objeto de entrada no esté vacío.
  .min(1).message('Se requiere al menos una clave de configuración para actualizar.');

export const analiticasSchema = Joi.object({
  fecha_inicio: Joi.date().iso().optional().allow(''),
  fecha_fin: Joi.date().iso().when('fecha_inicio', {
    is: Joi.date().iso().required(),
    then: Joi.date().min(Joi.ref('fecha_inicio')).optional().allow(''),
    otherwise: Joi.optional().allow('')
  }),
});

export const rutasUsuariosSchema = Joi.object({
  fecha_inicio: Joi.date().iso().required().messages({
    'any.required': 'El parámetro fecha_inicio es requerido.'
  }),
  fecha_fin: Joi.date().iso().min(Joi.ref('fecha_inicio')).required().messages({
    'any.required': 'El parámetro fecha_fin es requerido.',
    'date.min': 'La fecha_fin no puede ser anterior a la fecha_inicio.'
  })
});

export const resetPasswordSchema = Joi.object({
  password: Joi.string().min(6).required().messages({
    'string.min': 'La contraseña debe tener al menos 6 caracteres.',
    'any.required': 'El campo de contraseña es requerido.'
  })
});

export const crearAsignacionSchema = Joi.object({
  usuario_id: Joi.number().integer().min(1).required(),
  tipo_tarea: Joi.string().min(3).max(50).required(),
  referencia_tarea: Joi.string().min(1).max(100).required(),
  descripcion: Joi.string().allow('').max(500).optional(),
  fecha_limite: Joi.date().iso().optional().allow(null, '')
});

export const updateAsignacionSchema = Joi.object({
  usuario_id: Joi.number().integer().min(1).required(),
  tipo_tarea: Joi.string().min(3).max(50).required(),
  referencia_tarea: Joi.string().min(1).max(100).required(),
  descripcion: Joi.string().allow('').max(500).optional(),
  fecha_limite: Joi.date().iso().optional().allow(null, '')
});

export const updateEstadoAsignacionSchema = Joi.object({
  estado: Joi.string().valid('pendiente', 'en_proceso', 'completada', 'cancelada').required()
});

export const asignacionesFiltroSchema = Joi.object({
  estado: Joi.string().valid('pendiente', 'en_proceso', 'completada', 'cancelada').optional().allow(''),
  usuario_id: Joi.number().integer().min(1).optional().allow(''),
  page: Joi.number().integer().min(1).optional(),
  limit: Joi.number().integer().min(1).optional(),
  sortBy: Joi.string().valid('id', 'tipo_tarea', 'estado', 'fecha_limite', 'nombre_usuario', 'fecha_asignacion').optional(),
  sortOrder: Joi.string().valid('asc', 'desc').optional()
});

export const auditLogSchema = Joi.object({
  admin_id: Joi.number().integer().min(1).optional(),
  accion: Joi.string().optional(),
  page: Joi.number().integer().min(1).optional(),
  limit: Joi.number().integer().min(1).optional()
});