-- ===============================================
-- VISTAS COMPLETAS MEDIAPP 2025
-- ===============================================

DROP VIEW IF EXISTS vista_coordenadas,
vista_estadisticas_diarias,
vista_resumen_actividades,
vista_pendientes_sincronizacion,
vista_usuarios_detallada,
vista_medidores_detallada,
vista_cortes_detallada,
vista_actividad_general,
vista_tareas_asignadas,
vista_productividad_usuario,
vista_estadisticas_mensuales,
vista_estadisticas_diarias_dashboard,
vista_rendimiento_general,
vista_porcentaje_sincronizacion,
vista_alertas_operativas,
vista_alertas_detalle,
vista_usuarios_mas_productivos,
vista_usuarios_menos_productivos,
vista_actividad_georreferenciada,
vista_productividad_dashboard,
vista_rendimiento_tiempo,
vista_registros_por_dia_semana;

CREATE OR REPLACE VIEW vista_coordenadas AS
SELECT
    'cortes_servicio' AS tipo_registro,
    cs.id,
    cs.numero_cuenta AS referencia,
    ST_X(cs.ubicacion) AS longitud,
    ST_Y(cs.ubicacion) AS latitud,
    cs.fecha_registro,
    cs.usuario_id,
    NULL AS observaciones,
    cs.foto_procedimiento_path AS foto_path
FROM cortes_servicio cs
UNION ALL
SELECT 'entregas_individuales', ei.id, ei.cuenta, ST_X(ei.ubicacion), ST_Y(ei.ubicacion), ei.fecha_registro, ei.usuario_id, NULL, ei.foto_path
FROM entregas_individuales ei
UNION ALL
SELECT 'medidores', m.id, m.cuenta, ST_X(m.ubicacion), ST_Y(m.ubicacion), m.fecha_registro, m.usuario_id, NULL, m.foto_fachada_path
FROM medidores m
UNION ALL
SELECT 'notificadores', n.id, n.observaciones, ST_X(n.ubicacion), ST_Y(n.ubicacion), n.fecha_registro, n.usuario_id, n.observaciones, n.foto_path
FROM notificadores n
UNION ALL
SELECT 'entregas_edificio', ee.id, CONCAT(
        'Edificio - ', ee.total_cuentas, ' cuentas'
    ), ST_X(ee.ubicacion), ST_Y(ee.ubicacion), ee.fecha_registro, ee.usuario_id, NULL, ee.foto_path
FROM entregas_edificio ee
UNION ALL
SELECT 'infraestructura', i.id, 'Trabajo de infraestructura', ST_X(i.ubicacion), ST_Y(i.ubicacion), i.fecha_registro, i.usuario_id, i.descripcion, i.foto_factura_path
FROM infraestructura i;

CREATE OR REPLACE VIEW vista_estadisticas_diarias AS
SELECT
    DATE(vc.fecha_registro) AS fecha,
    COUNT(*) AS total_registros,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'cortes_servicio' THEN 1
            ELSE 0
        END
    ) AS cortes,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'medidores' THEN 1
            ELSE 0
        END
    ) AS medidores,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'entregas_individuales' THEN 1
            ELSE 0
        END
    ) AS entregas_individuales,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'notificadores' THEN 1
            ELSE 0
        END
    ) AS notificadores
FROM vista_coordenadas vc
GROUP BY
    fecha
ORDER BY fecha DESC;

CREATE OR REPLACE VIEW vista_resumen_actividades AS
SELECT
    u.id AS usuario_id,
    u.usuario,
    CONCAT(
        u.nombre,
        ' ',
        u.apellido_paterno
    ) AS nombre_completo,
    COUNT(DISTINCT cs.id) AS total_cortes,
    COUNT(DISTINCT ei.id) AS total_entregas_individuales,
    COUNT(DISTINCT ee.id) AS total_entregas_edificio,
    COUNT(DISTINCT i.id) AS total_infraestructuras,
    COUNT(DISTINCT m.id) AS total_medidores,
    COUNT(DISTINCT n.id) AS total_notificadores,
    MAX(cs.fecha_registro) AS ultimo_corte,
    MAX(m.fecha_lectura) AS ultima_lectura
FROM
    usuarios u
    LEFT JOIN cortes_servicio cs ON u.id = cs.usuario_id
    LEFT JOIN entregas_individuales ei ON u.id = ei.usuario_id
    LEFT JOIN entregas_edificio ee ON u.id = ee.usuario_id
    LEFT JOIN infraestructura i ON u.id = i.usuario_id
    LEFT JOIN medidores m ON u.id = m.usuario_id
    LEFT JOIN notificadores n ON u.id = n.usuario_id
GROUP BY
    u.id;

CREATE OR REPLACE VIEW vista_pendientes_sincronizacion AS
SELECT 'cortes_servicio' AS tipo_registro, cs.id, cs.usuario_id
FROM cortes_servicio cs
WHERE
    cs.sincronizado = 0
UNION ALL
SELECT 'entregas_individuales', ei.id, ei.usuario_id
FROM entregas_individuales ei
WHERE
    ei.sincronizado = 0
UNION ALL
SELECT 'entregas_edificio', ee.id, ee.usuario_id
FROM entregas_edificio ee
WHERE
    ee.sincronizado = 0
UNION ALL
SELECT 'infraestructura', i.id, i.usuario_id
FROM infraestructura i
WHERE
    i.sincronizado = 0
UNION ALL
SELECT 'medidores', m.id, m.usuario_id
FROM medidores m
WHERE
    m.sincronizado = 0
UNION ALL
SELECT 'notificadores', n.id, n.usuario_id
FROM notificadores n
WHERE
    n.sincronizado = 0;

CREATE OR REPLACE VIEW vista_usuarios_detallada AS
SELECT
    u.id,
    u.nombre,
    u.apellido_paterno,
    u.apellido_materno,
    CONCAT(
        u.nombre,
        ' ',
        u.apellido_paterno
    ) AS nombre_completo,
    u.numero_empleado,
    u.usuario,
    u.estatus_id,
    u.fecha_registro,
    u.ultimo_login,
    u.rol_id,
    r.nombre AS rol
FROM usuarios u
    LEFT JOIN roles r ON u.rol_id = r.id;

CREATE OR REPLACE VIEW vista_medidores_detallada AS
SELECT
    m.id,
    m.cuenta,
    m.medidor,
    m.lectura,
    ST_Y(m.ubicacion) AS latitud,
    ST_X(m.ubicacion) AS longitud,
    m.fecha_registro,
    u.usuario AS usuario_nombre,
    CONCAT(
        u.nombre,
        ' ',
        u.apellido_paterno
    ) AS nombre_completo,
    r.nombre AS rol
FROM
    medidores m
    LEFT JOIN usuarios u ON m.usuario_id = u.id
    LEFT JOIN roles r ON u.rol_id = r.id;

CREATE OR REPLACE VIEW vista_cortes_detallada AS
SELECT
    cs.id,
    cs.numero_cuenta,
    cs.numero_medidor,
    cs.suspension_agua,
    cs.suspension_drenaje,
    cs.precision_gps,
    ST_Y(cs.ubicacion) AS latitud,
    ST_X(cs.ubicacion) AS longitud,
    cs.fecha_registro,
    cs.sincronizado,
    cs.intentos_sincronizacion,
    u.usuario AS usuario_nombre,
    CONCAT(
        u.nombre,
        ' ',
        u.apellido_paterno
    ) AS nombre_completo
FROM
    cortes_servicio cs
    LEFT JOIN usuarios u ON cs.usuario_id = u.id;

CREATE OR REPLACE VIEW vista_actividad_general AS
SELECT
    vc.tipo_registro,
    vc.id,
    vc.referencia,
    vc.longitud,
    vc.latitud,
    vc.fecha_registro,
    u.usuario AS usuario_nombre,
    CONCAT(
        u.nombre,
        ' ',
        u.apellido_paterno
    ) AS nombre_completo,
    r.nombre AS rol
FROM
    vista_coordenadas vc
    LEFT JOIN usuarios u ON vc.usuario_id = u.id
    LEFT JOIN roles r ON u.rol_id = r.id;

CREATE OR REPLACE VIEW vista_estadisticas_mensuales AS
SELECT
    DATE_FORMAT(vc.fecha_registro, '%Y-%m') AS mes,
    COUNT(*) AS total_registros,
    COUNT(DISTINCT vc.usuario_id) AS usuarios_activos,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'medidores' THEN 1
            ELSE 0
        END
    ) AS total_medidores,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'cortes_servicio' THEN 1
            ELSE 0
        END
    ) AS total_cortes,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'notificadores' THEN 1
            ELSE 0
        END
    ) AS total_notificadores,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'entregas_individuales' THEN 1
            ELSE 0
        END
    ) AS total_entregas
FROM vista_coordenadas vc
GROUP BY
    mes
ORDER BY mes DESC;

CREATE OR REPLACE VIEW vista_estadisticas_diarias_dashboard AS
SELECT
    DATE(vc.fecha_registro) AS fecha,
    COUNT(*) AS total_registros,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'cortes_servicio' THEN 1
            ELSE 0
        END
    ) AS total_cortes,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'medidores' THEN 1
            ELSE 0
        END
    ) AS total_medidores,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'entregas_individuales' THEN 1
            ELSE 0
        END
    ) AS total_entregas,
    SUM(
        CASE
            WHEN vc.tipo_registro = 'notificadores' THEN 1
            ELSE 0
        END
    ) AS total_notificadores
FROM vista_coordenadas vc
GROUP BY
    fecha
ORDER BY fecha DESC;

CREATE OR REPLACE VIEW vista_rendimiento_general AS
SELECT
    COUNT(*) AS total_registros,
    COUNT(DISTINCT usuario_id) AS total_usuarios,
    ROUND(
        COUNT(*) / COUNT(DISTINCT usuario_id),
        2
    ) AS promedio_por_usuario,
    COUNT(DISTINCT DATE(fecha_registro)) AS dias_registro
FROM vista_coordenadas;

CREATE OR REPLACE VIEW vista_porcentaje_sincronizacion AS
SELECT 'cortes_servicio' AS tipo, ROUND(
        (
            SUM(
                CASE
                    WHEN sincronizado = 1 THEN 1
                    ELSE 0
                END
            ) / COUNT(*)
        ) * 100, 2
    ) AS porcentaje
FROM cortes_servicio
UNION ALL
SELECT 'medidores', ROUND(
        (
            SUM(
                CASE
                    WHEN sincronizado = 1 THEN 1
                    ELSE 0
                END
            ) / COUNT(*)
        ) * 100, 2
    )
FROM medidores
UNION ALL
SELECT 'notificadores', ROUND(
        (
            SUM(
                CASE
                    WHEN sincronizado = 1 THEN 1
                    ELSE 0
                END
            ) / COUNT(*)
        ) * 100, 2
    )
FROM notificadores
UNION ALL
SELECT 'entregas_individuales', ROUND(
        (
            SUM(
                CASE
                    WHEN sincronizado = 1 THEN 1
                    ELSE 0
                END
            ) / COUNT(*)
        ) * 100, 2
    )
FROM entregas_individuales;

CREATE OR REPLACE VIEW vista_alertas_operativas AS
SELECT
    u.usuario,
    COUNT(vp.id) AS pendientes,
    IF(
        COUNT(vp.id) > 50,
        'ALTA',
        IF(
            COUNT(vp.id) > 20,
            'MEDIA',
            'BAJA'
        )
    ) AS prioridad_alerta
FROM
    vista_pendientes_sincronizacion vp
    LEFT JOIN usuarios u ON u.id = vp.usuario_id
GROUP BY
    u.usuario
ORDER BY pendientes DESC;

CREATE OR REPLACE VIEW vista_alertas_detalle AS
SELECT u.usuario, vp.tipo_registro, COUNT(vp.id) AS total_pendientes
FROM
    vista_pendientes_sincronizacion vp
    LEFT JOIN usuarios u ON u.id = vp.usuario_id
GROUP BY
    u.usuario,
    vp.tipo_registro
ORDER BY u.usuario, total_pendientes DESC;

CREATE OR REPLACE VIEW vista_usuarios_mas_productivos AS
SELECT u.usuario, COUNT(vc.id) AS total_registros
FROM
    vista_coordenadas vc
    LEFT JOIN usuarios u ON vc.usuario_id = u.id
GROUP BY
    u.usuario
ORDER BY total_registros DESC
LIMIT 10;

CREATE OR REPLACE VIEW vista_usuarios_menos_productivos AS
SELECT u.usuario, COUNT(vc.id) AS total_registros_mes
FROM
    usuarios u
    LEFT JOIN vista_coordenadas vc ON u.id = vc.usuario_id
    AND vc.fecha_registro >= DATE_SUB(NOW(), INTERVAL 30 DAY)
GROUP BY
    u.usuario
ORDER BY total_registros_mes ASC
LIMIT 10;

CREATE OR REPLACE VIEW vista_actividad_georreferenciada AS
SELECT u.usuario, vc.tipo_registro, vc.longitud, vc.latitud, vc.fecha_registro
FROM
    vista_coordenadas vc
    INNER JOIN (
        SELECT usuario_id, MAX(fecha_registro) AS ultima_fecha
        FROM vista_coordenadas
        GROUP BY
            usuario_id
    ) ult ON ult.usuario_id = vc.usuario_id
    AND ult.ultima_fecha = vc.fecha_registro
    LEFT JOIN usuarios u ON vc.usuario_id = u.id;

CREATE OR REPLACE VIEW vista_productividad_dashboard AS
SELECT
    r.usuario_id,
    u.usuario,
    r.total_cortes + r.total_entregas_individuales + r.total_medidores + r.total_notificadores + r.total_entregas_edificio AS total_actividades,
    ROUND(
        (
            r.total_cortes + r.total_entregas_individuales + r.total_medidores + r.total_notificadores + r.total_entregas_edificio
        ) / NULLIF(
            DATEDIFF(NOW(), u.fecha_registro),
            0
        ),
        2
    ) AS promedio_diario,
    r.total_cortes,
    r.total_medidores,
    r.total_notificadores,
    ro.nombre AS rol
FROM
    vista_resumen_actividades r
    LEFT JOIN usuarios u ON u.id = r.usuario_id
    LEFT JOIN roles ro ON u.rol_id = ro.id;

CREATE OR REPLACE VIEW vista_rendimiento_tiempo AS
SELECT HOUR(vc.fecha_registro) AS hora, COUNT(vc.id) AS total_registros
FROM vista_coordenadas vc
GROUP BY
    hora
ORDER BY hora;

CREATE OR REPLACE VIEW vista_registros_por_dia_semana AS
SELECT DAYNAME(vc.fecha_registro) AS dia_semana, COUNT(vc.id) AS total_registros
FROM vista_coordenadas vc
GROUP BY
    dia_semana
ORDER BY FIELD(
        dia_semana, 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'
    );