import Joi from 'joi';

// --- EXTENSIÓN DE JOI PARA MANEJAR UBICACIÓN ---
// Esta extensión personalizada permite que los esquemas acepten un objeto `ubicacion`
// con `latitud` y `longitud`, y lo transforma en campos de primer nivel.
const joiWithUbicacion = Joi.extend((joi) => ({
  type: 'object',
  base: joi.object(),
  rules: {
    handleUbicacion: {
      method(enabled = true) {
        if (!enabled) {
          return this;
        }
        return this.custom((value, helpers) => {
          if (value.ubicacion && typeof value.ubicacion === 'object' && 'latitud' in value.ubicacion && 'longitud' in value.ubicacion) {
            // Si existe el objeto `ubicacion`, extrae latitud y longitud
            // y los pone en el nivel superior del objeto que se está validando.
            value.latitud = value.ubicacion.latitud;
            value.longitud = value.ubicacion.longitud;
            // Elimina el objeto `ubicacion` para que no interfiera con el resto de la validación.
            delete value.ubicacion;
          }
          return value; // Devuelve el objeto modificado (o el original).
        }, 'Manejo de ubicación');
      }
    }
  }
}));

// --- MIDDLEWARE GENERADOR DE VALIDACIÓN ---
// Esta es una "función de orden superior" que crea y devuelve un middleware de Express.
// Recibe un `schema` de Joi como argumento.
export const validate = (schema, source = 'body') => (req, res, next) => {
  let dataToValidate;
  switch (source) {
    case 'params':
      dataToValidate = req.params;
      break;
    case 'query':
      dataToValidate = req.query;
      break;
    case 'body':
    default:
      dataToValidate = req.body;
      break;
  }

  const { error, value } = schema.validate(dataToValidate, { allowUnknown: true });

  // Si `error` existe, significa que la validación falló.
  if (error) {
    // Se construye un mensaje de error claro a partir de los detalles del error de Joi.
    const errorMessage = error.details.map(detail => detail.message).join(', ');
    // Se responde con un código 400 (Bad Request) y el mensaje de error.
    return res.status(400).json({ error: `Datos inválidos en ${source}: ${errorMessage}` });
  }

  // Reemplaza la propiedad original de req con el valor validado y potencialmente transformado.
  switch (source) {
    case 'params': req.params = value; break;
    case 'query': req.query = value; break;
    case 'body': default: req.body = value; break;
  }
  // Si la validación es exitosa, se llama a `next()` para continuar con el siguiente middleware o controlador.
  next();
};