// --- MIDDLEWARE DE MANEJO DE ERRORES ---
// Este middleware especial de Express tiene 4 argumentos (err, req, res, next).
// Se activa cuando se llama a `next(error)` o cuando un error es lanzado
// en una ruta asíncrona (gracias a `express-async-errors`).
export const errorHandler = (err, req, res, next) => {
  // Siempre es importante registrar el error completo en la consola del servidor
  // para poder depurar problemas. Esto no se envía al cliente.
  console.error(err);

  // --- MANEJO DE ERRORES ESPECÍFICOS ---
  // Aquí puedes manejar errores conocidos para dar respuestas más claras.

  // Error de entrada duplicada de MySQL (ej. un usuario o email que ya existe).
  if (err.code === 'ER_DUP_ENTRY') {
    // 409 Conflict es el código HTTP apropiado para este caso.
    return res.status(409).json({ error: 'El registro ya existe. (Usuario o número de empleado duplicado)' });
  }

  // --- MANEJO DE ERRORES GENÉRICOS ---
  // Determina el código de estado HTTP. Si el error tiene una propiedad `status`
  // (que podemos asignar manualmente, ej. `err.status = 400`), la usamos.
  // Si no, usamos 500 (Error Interno del Servidor) como valor por defecto.
  const statusCode = err.status || 500; // Usar el status del error o 500 por defecto

  // Para proteger la aplicación, en producción no se deben filtrar detalles de errores 500.
  const isProduction = process.env.NODE_ENV === 'production';
  const errorMessage = isProduction && statusCode === 500 ? 'Ocurrió un error inesperado en el servidor.' : err.message;

  // Envía la respuesta de error final al cliente en formato JSON.
  res.status(statusCode).json({ error: errorMessage });
};