import { ROLES } from '../config/constants.js';

/**
 * Middleware para verificar si el usuario tiene el rol de Administrador.
 * Asume que el middleware `verificarToken` ya se ha ejecutado y ha
 * adjuntado el payload del token a `req.user`.
 */
export const verificarAdmin = (req, res, next) => {
  // El payload del token JWT contiene el nombre del rol como un string.  
  // Se permite el acceso si el rol es 'admin' o 'jefe de area'.
  const rolesPermitidos = ['admin', 'jefe de area'];
  if (req.user && rolesPermitidos.includes(req.user.rol)) {
    return next(); // El usuario es admin, puede continuar.
  }

  // Si no es admin, denegar el acceso.
  return res.status(403).json({ error: 'Acceso denegado. Se requiere rol de Administrador o Jefe de Área.' });
};

/**
 * Middleware genérico para verificar si el usuario tiene uno de los roles permitidos.
 * @param {Array<string>} rolesPermitidos - Un array con los nombres de los roles permitidos (ej. ['admin', 'jefe de area']).
 */
export const verificarRol = (rolesPermitidos) => {
  return (req, res, next) => {
    // Primero, asegúrate de que el usuario y su rol existan en la petición.
    if (!req.user || !req.user.rol) {
      return res.status(403).json({ error: 'Acceso denegado. No se pudo verificar el rol del usuario.' });
    }

    // Comprueba si el rol del usuario está incluido en la lista de roles permitidos.
    if (rolesPermitidos.includes(req.user.rol)) {
      return next(); // El rol es válido, permite el acceso.
    }

    // Si el rol no está en la lista, deniega el acceso.
    return res.status(403).json({ error: `Acceso denegado. Se requiere uno de los siguientes roles: ${rolesPermitidos.join(', ')}.` });
  };
};