// --- IMPORTACIONES DE MÓDULOS --- //
import express from 'express';
import path from 'path';
import { fileURLToPath } from 'url';
import cors from 'cors';
import { dirname } from 'path';
// Importaciones para Swagger
import swaggerUi from 'swagger-ui-express';import { swaggerSpec } from './config/swagger.js';

import bodyParser from 'body-parser';

// --- RUTAS Y MIDDLEWARES PERSONALIZADOS --- //
import allRoutes from './routes/index.js';
// Importa express-async-errors para capturar errores en rutas asíncronas automáticamente.
// Debe importarse antes que las rutas.
import 'express-async-errors';
import { errorHandler } from './middlewares/errorHandler.js';

// --- VALIDACIÓN DE VARIABLES DE ENTORNO CRÍTICAS --- //
if (!process.env.JWT_SECRET) {
  console.error('[FATAL] La variable JWT_SECRET no está definida. Abortando...');
  process.exit(1);
}

// --- CONFIGURACIÓN DE __dirname --- //
const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// --- CREACIÓN DE LA APLICACIÓN EXPRESS --- //
const app = express();

// --- MIDDLEWARES --- //
// Se flexibiliza CORS para permitir cualquier origen.
// Esto es especialmente útil para el handshake de WebSockets y el desarrollo.
// En producción, podrías querer una configuración más estricta, pero
// para WebSockets, a menudo la validación se hace en el evento 'connection'.
app.use(cors());

app.use(bodyParser.json());

// --- RUTAS PRINCIPALES DE LA APP --- //
// Se montan todas las rutas bajo el prefijo /api.
app.use('/api', allRoutes);
app.use('/api/uploads', express.static(path.join(__dirname, '..', 'uploads')));

// --- RUTA PARA LA DOCUMENTACIÓN DE LA API --- //
app.use('/api-docs', swaggerUi.serve, swaggerUi.setup(swaggerSpec));

// --- MANEJO CENTRALIZADO DE ERRORES --- //
app.use(errorHandler);

export default app;
