import { pool } from '../config/db.js';

/**
 * Calcula la distancia entre dos puntos geográficos usando la fórmula de Haversine.
 * @param {object} coords1 - { latitud, longitud } del primer punto.
 * @param {object} coords2 - { latitud, longitud } del segundo punto.
 * @returns {number} - La distancia en kilómetros.
 */
function haversineDistance(coords1, coords2) {
  const toRad = (x) => (x * Math.PI) / 180;
  const R = 6371; // Radio de la Tierra en km.

  const dLat = toRad(coords2.latitud - coords1.latitud);
  const dLon = toRad(coords2.longitud - coords1.longitud);
  const lat1 = toRad(coords1.latitud);
  const lat2 = toRad(coords2.latitud);

  const a =
    Math.sin(dLat / 2) * Math.sin(dLat / 2) +
    Math.sin(dLon / 2) * Math.sin(dLon / 2) * Math.cos(lat1) * Math.cos(lat2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

  return R * c;
}

export const fetchResumenActividades = async (usuarioId) => {
  const [rows] = await pool.execute(
    'SELECT * FROM vista_resumen_actividades WHERE usuario_id = ?',
    [usuarioId]
  );
  return rows[0] || {};
};

export const fetchEstadisticasDiarias = async (usuarioId) => {
  const [rows] = await pool.execute(
    `SELECT 
      CAST(fecha_registro AS DATE) as fecha, 
      tipo_registro as tipo, 
      COUNT(*) as cantidad 
     FROM vista_coordenadas 
     WHERE usuario_id = ? 
     GROUP BY fecha, tipo 
     ORDER BY fecha DESC`,
    [usuarioId]
  );
  return rows;
};

export const fetchCoordenadas = async (filters) => {
  const { tipo_registro, fecha_inicio, fecha_fin } = filters;

  let query = 'SELECT latitud, longitud FROM vista_coordenadas WHERE latitud IS NOT NULL AND longitud IS NOT NULL';
  const params = [];

  if (tipo_registro) {
    query += ' AND tipo_registro = ?';
    params.push(tipo_registro);
  }

  if (fecha_inicio && fecha_fin) {
    query += ' AND fecha_registro BETWEEN ? AND ?';
    params.push(fecha_inicio, fecha_fin);
  }

  const [rows] = await pool.execute(query, params);
  return rows;
};

export const fetchHistorialUsuario = async (usuarioId, filters) => {
  const { tipo_registro, page = 1, limit = 20 } = filters;

  const pageNum = parseInt(page, 10) || 1;
  const limitNum = parseInt(limit, 10) || 20;
  const offset = (pageNum - 1) * limitNum;

  let baseQuery = 'FROM vista_coordenadas WHERE usuario_id = ?';
  const params = [usuarioId];

  if (tipo_registro) {
    baseQuery += ' AND tipo_registro = ?';
    params.push(tipo_registro);
  }

  const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
  const [countResult] = await pool.execute(countQuery, params);
  const total = countResult[0].total;

  const dataQuery = `SELECT * ${baseQuery} ORDER BY fecha_registro DESC LIMIT ? OFFSET ?`;
  const dataParams = [...params, limitNum, offset];
  const [rows] = await pool.execute(dataQuery, dataParams);

  return {
    data: rows,
    pagination: {
      total,
      page: pageNum,
      limit: limitNum,
      totalPages: Math.ceil(total / limitNum),
      hasNextPage: pageNum * limitNum < total,
      hasPrevPage: pageNum > 1,
    },
  };
};

export const fetchRegistrosCercanos = async (usuarioId, geoParams) => {
  const { latitud, longitud, radio } = geoParams;
  const [results] = await pool.execute(
    'CALL sp_buscar_registros_cercanos(?, ?, ?, ?)',
    [latitud, longitud, radio, usuarioId]
  );

  return results[0].map(registro => ({
    ...registro,
    ubicacion: { coordinates: [registro.longitud, registro.latitud] }
  }));
};

export const fetchHistorialUbicacionesPorFecha = async (usuarioId, dateFilters) => {
  const { fecha_inicio, fecha_fin } = dateFilters;
  const [rows] = await pool.execute(
    `SELECT 
       ST_Y(ubicacion) as latitud, 
       ST_X(ubicacion) as longitud, 
       timestamp 
     FROM historial_ubicaciones 
     WHERE usuario_id = ? AND timestamp BETWEEN ? AND ? 
     ORDER BY timestamp ASC`,
    [usuarioId, fecha_inicio, fecha_fin]
  );

  let distanciaTotalKm = 0;
  if (rows.length > 1) {
    for (let i = 1; i < rows.length; i++) {
      distanciaTotalKm += haversineDistance(rows[i - 1], rows[i]);
    }
  }

  return {
    ruta: rows,
    distancia_total_km: parseFloat(distanciaTotalKm.toFixed(2))
  };
};