import { pool } from '../config/db.js';

/**
 * Registra una acción de administrador en el log de auditoría.
 * @param {number} adminId - ID del administrador que realiza la acción.
 * @param {string} accion - Nombre de la acción (ej. 'DELETE_RECORD').
 * @param {object} [detalles={}] - Objeto con detalles adicionales.
 * @param {string} [detalles.recurso_afectado] - La tabla o recurso afectado.
 * @param {number} [detalles.recurso_id] - El ID del recurso afectado.
 * @param {object} [detalles.cambios] - Objeto con los cambios realizados.
 */
export const logAdminAction = async (adminId, accion, detalles = {}) => {
  const { recurso_afectado, recurso_id, cambios } = detalles;
  try {
    await pool.execute(
      `INSERT INTO log_acciones_admin (admin_id, accion, recurso_afectado, recurso_id, detalles)
       VALUES (?, ?, ?, ?, ?)`,
      [
        adminId,
        accion,
        recurso_afectado || null,
        recurso_id || null,
        cambios ? JSON.stringify(cambios) : null,
      ]
    );
  } catch (error) {
    // Es importante que un fallo en el log no detenga la acción principal.
    console.error('Error al registrar la acción de auditoría:', error);
  }
};

/**
 * Obtiene el log de auditoría de los administradores.
 * @param {Object} filters - Filtros de búsqueda y paginación.
 */
export const fetchAuditLog = async (filters = {}) => {
  const { admin_id, accion, page = 1, limit = 20 } = filters;
  const offset = (page - 1) * limit;

  let baseQuery = `FROM log_acciones_admin l JOIN usuarios u ON l.admin_id = u.id`;
  const whereClauses = [];
  const params = [];

  if (admin_id) {
    whereClauses.push('l.admin_id = ?');
    params.push(admin_id);
  }
  if (accion) {
    whereClauses.push('l.accion = ?');
    params.push(accion);
  }

  if (whereClauses.length > 0) {
    baseQuery += ` WHERE ${whereClauses.join(' AND ')}`;
  }

  const countQuery = `SELECT COUNT(l.id) as total ${baseQuery}`;
  const [countResult] = await pool.execute(countQuery, params);
  const total = countResult[0].total;

  const dataQuery = `SELECT l.*, CONCAT(u.nombre, ' ', u.apellido_paterno) as admin_nombre ${baseQuery} ORDER BY l.fecha DESC LIMIT ? OFFSET ?`;
  const dataParams = [...params, limit, offset];
  const [rows] = await pool.execute(dataQuery, dataParams);

  return { data: rows, pagination: { total, page, limit, totalPages: Math.ceil(total / limit) } };
};