import { pool } from '../config/db.js';
import bcrypt from 'bcryptjs';
import { logAdminAction } from './audit.service.js';
import {
  PAGINATION,
  ALLOWED_TABLES,
  USER_STATUS,
  ADMIN_ACTIONS
} from '../config/constants.js';

/**
 * Formatea fecha para consultas SQL
 * @param {string} dateString - Fecha en formato string
 * @returns {string} Fecha formateada
 */
const formatDateForQuery = (dateString) => {
  return `${dateString} 23:59:59`;
};

/**
 * Valida rango de fechas
 * @param {string} start - Fecha inicio
 * @param {string} end - Fecha fin
 * @throws {Error} Si el rango es inválido
 */
const validateDateRange = (start, end) => {
  if (start && end && new Date(start) > new Date(end)) {
    throw new Error('La fecha de inicio no puede ser mayor a la fecha fin');
  }
};

/**
 * Procesa y valida los parámetros de paginación de forma segura.
 * @param {object} filters - Objeto con los filtros de la petición.
 * @returns {{pageNum: number, limitNum: number, offset: number}}
 */
const getSafePagination = (filters) => {
  let pageNum = parseInt(filters.page, 10);
  let limitNum = parseInt(filters.limit, 10);

  if (isNaN(pageNum) || pageNum < 1) {
    pageNum = PAGINATION.DEFAULT_PAGE;
  }

  if (isNaN(limitNum) || limitNum < 1) {
    limitNum = PAGINATION.DEFAULT_LIMIT;
  } else if (limitNum > PAGINATION.MAX_LIMIT) {
    limitNum = PAGINATION.MAX_LIMIT; // Clamping to max limit
  }

  const offset = (pageNum - 1) * limitNum;
  return { pageNum, limitNum, offset };
};


/**
 * Obtiene resumen global del sistema
 * @returns {Promise<Object>} Resumen con estadísticas
 */
export const fetchResumenGlobal = async () => {
  try {
    const [results] = await pool.execute(`
      SELECT
        (SELECT COUNT(*) FROM usuarios WHERE estatus_id = ?) as total_usuarios,
        (SELECT COUNT(*) FROM cortes_servicio) as total_cortes,
        (SELECT COUNT(*) FROM medidores) as total_lecturas,
        (SELECT COUNT(*) FROM notificadores) as total_notificaciones,
        ((SELECT COUNT(*) FROM entregas_individuales) + (SELECT COUNT(*) FROM entregas_edificio)) as total_entregas,
        (SELECT COUNT(*) FROM vista_coordenadas WHERE DATE(fecha_registro) = CURDATE()) as registros_hoy
    `, [USER_STATUS.ACTIVE]);

    const summary = results[0] || {};

    return {
      total_usuarios: summary.total_usuarios || 0,
      total_cortes: summary.total_cortes || 0,
      total_lecturas: summary.total_lecturas || 0,
      total_notificaciones: summary.total_notificaciones || 0,
      total_entregas: summary.total_entregas || 0,
      registros_hoy: summary.registros_hoy || 0
    };
  } catch (error) {
    console.error('Error en fetchResumenGlobal:', error);
    throw new Error('No se pudo obtener el resumen global');
  }
};

/**
 * Obtiene ranking de productividad de usuarios
 * @returns {Promise<Array>} Array con ranking de usuarios
 */
export const fetchRankingProductividad = async () => {
  try {
    const [rows] = await pool.execute(`
      SELECT 
        vra.*,
        (vra.total_cortes + vra.total_medidores + vra.total_notificadores + 
         vra.total_entregas_individuales + vra.total_entregas_edificio + 
         vra.total_infraestructuras) AS total_actividades,
        vc.latitud,
        vc.longitud
      FROM 
        vista_resumen_actividades vra
      LEFT JOIN 
        vista_coordenadas vc ON vc.id = (
          SELECT id 
          FROM vista_coordenadas 
          WHERE usuario_id = vra.usuario_id 
          ORDER BY fecha_registro DESC 
          LIMIT 1
        )
      WHERE 
        (vra.total_cortes + vra.total_medidores + vra.total_notificadores + 
         vra.total_entregas_individuales + vra.total_entregas_edificio + 
         vra.total_infraestructuras) > 0
      ORDER BY 
        total_actividades DESC
      LIMIT 5;
    `);
    return rows;
  } catch (error) {
    console.error('Error en fetchRankingProductividad:', error);
    throw new Error('No se pudo obtener el ranking de productividad');
  }
};

/**
 * Obtiene actividad reciente del sistema
 * @returns {Promise<Array>} Array con actividad reciente
 */
export const fetchActividadReciente = async () => {
  try {
    const [rows] = await pool.execute(`
      SELECT
        vc.id, vc.tipo_registro, vc.referencia, vc.fecha_registro as fecha,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as usuario_nombre
      FROM vista_coordenadas vc
      JOIN usuarios u ON vc.usuario_id = u.id
      ORDER BY vc.fecha_registro DESC
      LIMIT 15
    `);
    return rows;
  } catch (error) {
    console.error('Error en fetchActividadReciente:', error);
    throw new Error('No se pudo obtener la actividad reciente');
  }
};

/**
 * Obtiene la última ubicación registrada de cada usuario activo.
 * @returns {Promise<Array>}
 */
export const fetchUbicacionesEnVivo = async () => {
  try {
    const [rows] = await pool.execute(`
      SELECT u.id, CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_completo, vc.latitud, vc.longitud, vc.fecha_registro
      FROM usuarios u
      JOIN (
          SELECT usuario_id, MAX(id) as max_id
          FROM vista_coordenadas
          GROUP BY usuario_id
      ) AS ultimos_registros ON u.id = ultimos_registros.usuario_id
      JOIN vista_coordenadas vc ON ultimos_registros.max_id = vc.id
      WHERE u.estatus_id = ? AND vc.latitud IS NOT NULL AND vc.longitud IS NOT NULL;
    `, [USER_STATUS.ACTIVE]);
    return rows;
  } catch (error) {
    console.error('Error en fetchUbicacionesEnVivo:', error);
    throw new Error('No se pudieron obtener las ubicaciones en vivo.');
  }
};

/**
 * Obtiene historial global con filtros y paginación
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchHistorialGlobalFiltrado = async (filters) => {
  try {
    const { nombre_usuario, usuario_id, tipo_registro, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM vista_coordenadas vc JOIN usuarios u ON vc.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (nombre_usuario) {
      baseQuery += " AND CONCAT(u.nombre, ' ', u.apellido_paterno, ' ', u.apellido_materno) LIKE ?";
      params.push(`%${nombre_usuario}%`);
    }
    if (usuario_id) {
      baseQuery += ' AND usuario_id = ?';
      params.push(usuario_id);
    }
    if (tipo_registro) {
      baseQuery += ' AND tipo_registro = ?';
      params.push(tipo_registro);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND vc.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `SELECT vc.*, CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario ${baseQuery} ORDER BY vc.fecha_registro DESC LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchHistorialGlobalFiltrado:', error);
    throw error; // Relanzar error para manejo específico
  }
};

/**
 * Obtiene estado de sincronización de usuarios
 * @returns {Promise<Array>} Array con estado de sincronización
 */
export const fetchEstadoSincronizacion = async () => {
  try {
    const [rows] = await pool.execute(`
      SELECT 
        u.id as usuario_id, 
        u.nombre_completo,
        COUNT(vps.id) as pendientes_sincronizar,
        CASE 
          WHEN u.estatus_id = ? THEN 'activo' 
          ELSE 'inactivo' 
        END as estado
      FROM usuarios u
      LEFT JOIN vista_pendientes_sincronizacion vps ON vps.usuario_id = u.id
      WHERE u.estatus_id = ?
      GROUP BY u.id, u.nombre_completo, u.estatus_id
      ORDER BY pendientes_sincronizar DESC, nombre_completo ASC
    `, [USER_STATUS.ACTIVE, USER_STATUS.ACTIVE]);
    return rows;
  } catch (error) {
    console.error('Error en fetchEstadoSincronizacion:', error);
    throw new Error('No se pudo obtener el estado de sincronización');
  }
};

/**
 * Obtiene últimos registros por usuario usando consulta optimizada
 * @param {number} usuario_id - ID del usuario
 * @returns {Promise<Object>} Objeto con últimos registros
 */
export const fetchUltimosRegistrosPorUsuario = async (usuario_id) => {
  try {
    const query = `
      SELECT 'corte_servicio' as tipo, cs.* 
      FROM cortes_servicio cs 
      WHERE cs.usuario_id = ? 
      ORDER BY cs.fecha_registro DESC 
      LIMIT 1
      
      UNION ALL
      
      SELECT 'medidor' as tipo, m.* 
      FROM medidores m 
      WHERE m.usuario_id = ? 
      ORDER BY m.fecha_registro DESC 
      LIMIT 1
      
      UNION ALL
      
      SELECT 'notificador' as tipo, n.* 
      FROM notificadores n 
      WHERE n.usuario_id = ? 
      ORDER BY n.fecha_registro DESC 
      LIMIT 1
      
      UNION ALL
      
      SELECT 'entrega_individual' as tipo, ei.* 
      FROM entregas_individuales ei 
      WHERE ei.usuario_id = ? 
      ORDER BY ei.fecha_registro DESC 
      LIMIT 1
      
      UNION ALL
      
      SELECT 'entrega_edificio' as tipo, ee.* 
      FROM entregas_edificio ee 
      WHERE ee.usuario_id = ? 
      ORDER BY ee.fecha_registro DESC 
      LIMIT 1
      
      UNION ALL
      
      SELECT 'infraestructura' as tipo, i.* 
      FROM infraestructura i 
      WHERE i.usuario_id = ? 
      ORDER BY i.fecha_registro DESC 
      LIMIT 1
    `;

    const [rows] = await pool.execute(query,
      [usuario_id, usuario_id, usuario_id, usuario_id, usuario_id, usuario_id]
    );

    // Procesar resultados
    const result = {
      corte_servicio: null,
      medidor: null,
      notificador: null,
      entrega_individual: null,
      entrega_edificio: null,
      infraestructura: null
    };

    rows.forEach(row => {
      const { tipo, ...datos } = row;
      const key = tipo.replace('-', '_');
      result[key] = datos;
    });

    return result;
  } catch (error) {
    console.error('Error en fetchUltimosRegistrosPorUsuario:', error);
    throw new Error('No se pudieron obtener los últimos registros del usuario');
  }
};

/**
 * Obtiene lista de usuarios
 * @returns {Promise<Array>} Array de usuarios
 */
export const fetchUsuarios = async (filters = {}) => {
  try {
    const { page, limit, sortBy, sortOrder, role, term } = filters;
    const { pageNum, limitNum, offset } = getSafePagination({ page, limit });

    let baseQuery = 'FROM vista_usuarios_detallada WHERE 1=1';
    const params = [];

    if (role && role !== 'todos') {
      baseQuery += ' AND rol = ?';
      params.push(role);
    }

    if (term) {
      baseQuery += " AND (nombre_completo LIKE ? OR usuario LIKE ? OR numero_empleado LIKE ?)";
      params.push(`%${term}%`, `%${term}%`, `%${term}%`);
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const totalItems = countResult[0].total;

    const sortColumn = ['id', 'nombre_completo', 'rol'].includes(sortBy) ? sortBy : 'id';
    const order = sortOrder === 'desc' ? 'DESC' : 'ASC';

    const dataQuery = `SELECT * ${baseQuery} ORDER BY ${sortColumn} ${order} LIMIT ? OFFSET ?`;
    const [rows] = await pool.execute(dataQuery, [...params, limitNum, offset]);

    return {
      data: rows,
      pagination: {
        totalItems,
        totalPages: Math.ceil(totalItems / limitNum),
        currentPage: pageNum,
        limit: limitNum,
      },
    };
  } catch (error) {
    console.error('Error en fetchUsuarios:', error);
    throw new Error('No se pudo obtener la lista de usuarios');
  }
};

/**
 * Obtiene un perfil completo de un usuario para su página de detalles.
 * @param {number} userId - ID del usuario.
 * @returns {Promise<Object>}
 */
export const fetchUserProfile = async (userId) => {
  try {
    const [userPromise, summaryPromise, recentActivityPromise] = await Promise.all([
      // 1. Obtener detalles del usuario directamente de la tabla de usuarios y roles
      pool.execute(`
        SELECT u.*, r.nombre as rol 
        FROM usuarios u 
        LEFT JOIN roles r ON u.rol_id = r.id 
        WHERE u.id = ?`, [userId]),
      // 2. Obtener resumen de actividades del usuario
      pool.execute('SELECT * FROM vista_resumen_actividades WHERE usuario_id = ?', [userId]),
      // 3. Obtener últimas 5 actividades con ubicación
      pool.execute(`
        SELECT id, tipo_registro, referencia, fecha_registro, latitud, longitud 
        FROM vista_coordenadas 
        WHERE usuario_id = ? 
        ORDER BY fecha_registro DESC 
        LIMIT 5
      `, [userId])
    ]);

    const user = userPromise[0][0];
    if (!user) {
      const error = new Error('Usuario no encontrado');
      error.status = 404;
      throw error;
    }

    return {
      detalles: user,
      resumen_actividades: summaryPromise[0][0] || {},
      actividad_reciente: recentActivityPromise[0] || [],
    };
  } catch (error) {
    console.error(`Error en fetchUserProfile para el usuario ${userId}:`, error);
    throw error;
  }
};

/**
 * Actualiza estatus de usuario
 * @param {number} adminId - ID del administrador
 * @param {number} targetUserId - ID del usuario objetivo
 * @param {number} estatus_id - Nuevo estatus
 * @returns {Promise<Object>} Resultado de la operación
 */
export const updateUserStatus = async (adminId, targetUserId, estatus_id) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    const [result] = await conn.execute(
      'UPDATE usuarios SET estatus_id = ? WHERE id = ?',
      [estatus_id, targetUserId]
    );

    if (result.affectedRows > 0) {
      await logAdminAction(adminId, ADMIN_ACTIONS.UPDATE_USER_STATUS, {
        recurso_afectado: 'usuarios',
        recurso_id: targetUserId,
        cambios: { estatus_id }
      });
    }

    await conn.commit();
    return result;
  } catch (error) {
    await conn.rollback();
    console.error('Error en updateUserStatus:', error);
    throw new Error('No se pudo actualizar el estatus del usuario');
  } finally {
    conn.release();
  }
};

/**
 * Actualiza detalles de usuario
 * @param {number} adminId - ID del administrador
 * @param {number} targetUserId - ID del usuario objetivo
 * @param {Object} userData - Datos del usuario
 * @returns {Promise<Object>} Resultado de la operación
 */
export const updateUserDetails = async (adminId, targetUserId, userData) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    const { nombre, apellido_paterno, apellido_materno, numero_empleado, usuario, rol_id } = userData;

    const [result] = await conn.execute(
      'UPDATE usuarios SET nombre = ?, apellido_paterno = ?, apellido_materno = ?, numero_empleado = ?, usuario = ?, rol_id = ? WHERE id = ?',
      [nombre, apellido_paterno, apellido_materno, numero_empleado, usuario, rol_id, targetUserId]
    );

    if (result.affectedRows > 0) {
      await logAdminAction(adminId, ADMIN_ACTIONS.UPDATE_USER_DETAILS, {
        recurso_afectado: 'usuarios',
        recurso_id: targetUserId
      });
    }

    await conn.commit();
    return result;
  } catch (error) {
    await conn.rollback();
    console.error('Error en updateUserDetails:', error);
    throw new Error('No se pudieron actualizar los datos del usuario');
  } finally {
    conn.release();
  }
};

/**
 * Crea nuevo usuario
 * @param {number} adminId - ID del administrador
 * @param {Object} userData - Datos del nuevo usuario
 * @returns {Promise<Object>} Usuario creado
 */
export const createNewUser = async (adminId, userData) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    const { nombre, apellido_paterno, apellido_materno, numero_empleado, usuario, password, rol_id } = userData;

    // Validar que el usuario no exista
    const [existingUsers] = await conn.execute(
      'SELECT id FROM usuarios WHERE usuario = ?',
      [usuario]
    );

    if (existingUsers.length > 0) {
      throw new Error('El nombre de usuario ya existe');
    }

    const salt = await bcrypt.genSalt(10);
    const password_hash = await bcrypt.hash(password, salt);

    const [result] = await conn.execute(
      'INSERT INTO usuarios (nombre, apellido_paterno, apellido_materno, numero_empleado, usuario, password_hash, rol_id) VALUES (?, ?, ?, ?, ?, ?, ?)',
      [nombre, apellido_paterno, apellido_materno, numero_empleado, usuario, password_hash, rol_id || 3]
    );

    await logAdminAction(adminId, ADMIN_ACTIONS.CREATE_USER, {
      recurso_afectado: 'usuarios',
      recurso_id: result.insertId
    });

    await conn.commit();
    return { id: result.insertId, ...userData };
  } catch (error) {
    await conn.rollback();
    console.error('Error en createNewUser:', error);
    throw error; // Relanzar error para manejo específico
  } finally {
    conn.release();
  }
};

/**
 * Elimina registro de tabla permitida
 * @param {number} adminId - ID del administrador
 * @param {string} tabla - Nombre de la tabla
 * @param {number} id - ID del registro
 * @returns {Promise<Object>} Resultado de la operación
 */
export const deleteRecord = async (adminId, tabla, id) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    // Validar tabla permitida
    if (!ALLOWED_TABLES.includes(tabla)) {
      throw new Error('Operación no permitida en esta tabla');
    }

    const [result] = await conn.execute(
      `DELETE FROM ${tabla} WHERE id = ?`,
      [id]
    );

    if (result.affectedRows > 0) {
      await logAdminAction(adminId, ADMIN_ACTIONS.DELETE_RECORD, {
        recurso_afectado: tabla,
        recurso_id: id
      });
    }

    await conn.commit();
    return result;
  } catch (error) {
    await conn.rollback();
    console.error('Error en deleteRecord:', error);
    throw new Error('No se pudo eliminar el registro');
  } finally {
    conn.release();
  }
};

// Cache para configuraciones
let configCache = null;
let cacheTimestamp = null;
const CACHE_TTL = 5 * 60 * 1000; // 5 minutos

/**
 * Obtiene configuraciones del sistema
 * @returns {Promise<Object>} Objeto con configuraciones
 */
export const fetchConfiguraciones = async () => {
  try {
    const now = Date.now();

    if (configCache && cacheTimestamp && (now - cacheTimestamp) < CACHE_TTL) {
      return configCache;
    }

    const [results] = await pool.execute('SELECT clave, valor FROM configuraciones');

    configCache = results.reduce((acc, row) => {
      acc[row.clave] = row.valor;
      return acc;
    }, {});

    cacheTimestamp = now;
    return configCache;
  } catch (error) {
    console.error('Error en fetchConfiguraciones:', error);
    throw new Error('No se pudieron obtener las configuraciones');
  }
};

/**
 * Actualiza configuraciones del sistema
 * @param {number} adminId - ID del administrador
 * @param {Object} configs - Configuraciones a actualizar
 * @returns {Promise<void>}
 */
export const updateConfiguraciones = async (adminId, configs) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    const updatePromises = Object.entries(configs).map(([clave, valor]) => {
      return conn.execute(
        'UPDATE configuraciones SET valor = ? WHERE clave = ?',
        [valor.toString(), clave]
      );
    });

    await Promise.all(updatePromises);

    // Invalidar cache
    configCache = null;
    cacheTimestamp = null;

    await conn.commit();

    await logAdminAction(adminId, ADMIN_ACTIONS.UPDATE_CONFIGURATIONS, {
      cambios: configs
    });
  } catch (error) {
    await conn.rollback();
    console.error('Error en updateConfiguraciones:', error);
    throw new Error('No se pudieron actualizar las configuraciones');
  } finally {
    conn.release();
  }
};

// Cache para analíticas sin filtro de fecha
const analyticsCache = {
  actividadPorHora: { data: null, timestamp: null },
  registrosPorTipo: { data: null, timestamp: null },
  topUsuariosAsignaciones: { data: null, timestamp: null },
};
const ANALYTICS_CACHE_TTL = 60 * 1000; // 1 minuto

/**
 * Obtiene actividad por hora
 * @param {Object} filters - Filtros de fecha
 * @returns {Promise<Array>} Array con actividad por hora
 */
export const fetchActividadPorHora = async (filters) => {
  try {
    const now = Date.now();
    const hasFilters = filters.fecha_inicio && filters.fecha_fin;

    if (!hasFilters && analyticsCache.actividadPorHora.data && (now - analyticsCache.actividadPorHora.timestamp < ANALYTICS_CACHE_TTL)) {
      return analyticsCache.actividadPorHora.data;
    }

    const { fecha_inicio, fecha_fin } = filters;

    validateDateRange(fecha_inicio, fecha_fin);

    let query = `
      SELECT
        HOUR(fecha_registro) as hora,
        COUNT(*) as cantidad
      FROM vista_coordenadas
    `;
    const params = [];
    const whereClauses = [];

    if (fecha_inicio && fecha_fin) {
      whereClauses.push('fecha_registro BETWEEN ? AND ?');
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    if (whereClauses.length > 0) {
      query += ' WHERE ' + whereClauses.join(' AND ');
    }

    query += ' GROUP BY hora ORDER BY hora ASC';

    const [rows] = await pool.execute(query, params);

    if (!hasFilters) {
      analyticsCache.actividadPorHora = { data: rows, timestamp: now };
    }

    return rows;
  } catch (error) {
    console.error('Error en fetchActividadPorHora:', error);
    throw new Error('No se pudo obtener la actividad por hora');
  }
};

/**
 * Obtiene la actividad por día de la semana.
 * @param {Object} filters - Filtros de fecha.
 * @returns {Promise<Array>}
 */
export const fetchActividadPorDiaSemana = async (filters) => {
  try {
    const { fecha_inicio, fecha_fin } = filters;
    let query = `
      SELECT DAYNAME(fecha_registro) as dia, COUNT(*) as cantidad
      FROM vista_coordenadas
    `;
    const params = [];
    if (fecha_inicio && fecha_fin) {
      query += ' WHERE fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }
    query += ' GROUP BY dia ORDER BY FIELD(dia, "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday")';

    const [rows] = await pool.execute(query, params);
    return rows;
  } catch (error) {
    console.error('Error en fetchActividadPorDiaSemana:', error);
    throw new Error('No se pudo obtener la actividad por día de la semana.');
  }
};

/**
 * Obtiene el top 5 de usuarios con más asignaciones.
 * @returns {Promise<Array>}
 */
export const fetchTopUsuariosPorAsignaciones = async () => {
  try {
    const now = Date.now();
    if (analyticsCache.topUsuariosAsignaciones.data && (now - analyticsCache.topUsuariosAsignaciones.timestamp < ANALYTICS_CACHE_TTL)) {
      return analyticsCache.topUsuariosAsignaciones.data;
    }

    const [rows] = await pool.execute(`
      SELECT u.id, CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_completo, COUNT(a.id) as total_asignaciones
      FROM asignaciones_trabajo a
      JOIN usuarios u ON a.usuario_asignado_id = u.id
      GROUP BY u.id, nombre_completo
      ORDER BY total_asignaciones DESC
      LIMIT 5
    `);

    analyticsCache.topUsuariosAsignaciones = { data: rows, timestamp: now };

    return rows;
  } catch (error) {
    console.error('Error en fetchTopUsuariosPorAsignaciones:', error);
    throw new Error('No se pudo obtener el top de usuarios por asignaciones.');
  }
};

/**
 * Obtiene el historial de ubicaciones de todos los usuarios para un rango de fechas.
 * @param {Object} dateFilters - Filtros de fecha.
 * @returns {Promise<Array>} Lista de usuarios con sus rutas.
 */
export const fetchRutasUsuarios = async (dateFilters) => {
  const { fecha_inicio, fecha_fin } = dateFilters;
  if (!fecha_inicio || !fecha_fin) {
    throw new Error('Se requieren fecha de inicio y fin para buscar rutas.');
  }

  const [rows] = await pool.execute(
    `SELECT 
       hu.usuario_id,
       u.nombre,
       u.apellido_paterno,
       ST_Y(hu.ubicacion) as latitud, 
       ST_X(hu.ubicacion) as longitud, 
       hu.timestamp 
     FROM historial_ubicaciones hu
     JOIN usuarios u ON hu.usuario_id = u.id
     WHERE hu.timestamp BETWEEN ? AND ?
     ORDER BY hu.usuario_id, hu.timestamp ASC`,
    [fecha_inicio, `${fecha_fin} 23:59:59`]
  );
  return rows;
};

/**
 * Resetea contraseña de usuario
 * @param {number} adminId - ID del administrador
 * @param {number} targetUserId - ID del usuario objetivo
 * @param {string} newPassword - Nueva contraseña
 * @returns {Promise<Object>} Resultado de la operación
 */
export const resetUserPassword = async (adminId, targetUserId, newPassword) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    const salt = await bcrypt.genSalt(10);
    const password_hash = await bcrypt.hash(newPassword, salt);

    const [result] = await conn.execute(
      'UPDATE usuarios SET password_hash = ? WHERE id = ?',
      [password_hash, targetUserId]
    );

    if (result.affectedRows > 0) {
      await logAdminAction(adminId, ADMIN_ACTIONS.RESET_USER_PASSWORD, {
        recurso_afectado: 'usuarios',
        recurso_id: targetUserId
      });
    }

    await conn.commit();
    return result;
  } catch (error) {
    await conn.rollback();
    console.error('Error en resetUserPassword:', error);
    throw new Error('No se pudo resetear la contraseña');
  } finally {
    conn.release();
  }
};

/**
 * Obtiene registros por tipo
 * @param {Object} filters - Filtros de fecha
 * @returns {Promise<Array>} Array con registros por tipo
 */
export const fetchRegistrosPorTipo = async (filters) => {
  try {
    const now = Date.now();
    const hasFilters = filters.fecha_inicio && filters.fecha_fin;

    if (!hasFilters && analyticsCache.registrosPorTipo.data && (now - analyticsCache.registrosPorTipo.timestamp < ANALYTICS_CACHE_TTL)) {
      return analyticsCache.registrosPorTipo.data;
    }

    const { fecha_inicio, fecha_fin } = filters;

    validateDateRange(fecha_inicio, fecha_fin);

    let query = `
      SELECT
        tipo_registro,
        COUNT(*) as cantidad
      FROM vista_coordenadas
    `;
    const params = [];
    const whereClauses = [];

    if (fecha_inicio && fecha_fin) {
      whereClauses.push('fecha_registro BETWEEN ? AND ?');
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    if (whereClauses.length > 0) {
      query += ' WHERE ' + whereClauses.join(' AND ');
    }

    query += ' GROUP BY tipo_registro ORDER BY cantidad DESC';

    const [rows] = await pool.execute(query, params);

    if (!hasFilters) {
      analyticsCache.registrosPorTipo = { data: rows, timestamp: now };
    }

    return rows;
  } catch (error) {
    console.error('Error en fetchRegistrosPorTipo:', error);
    throw new Error('No se pudieron obtener los registros por tipo');
  }
};

/**
 * Obtiene registros detallados de cortes de servicio con filtros y paginación.
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchCortesServicioDetallado = async (filters) => {
  try {
    const { usuario_id, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM cortes_servicio cs JOIN usuarios u ON cs.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (usuario_id) {
      baseQuery += ' AND cs.usuario_id = ?';
      params.push(usuario_id);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND cs.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `
      SELECT
        cs.id, cs.numero_cuenta, cs.numero_medidor, cs.suspension_agua, cs.suspension_drenaje,
        cs.precision_gps, ST_Y(cs.ubicacion) as latitud, ST_X(cs.ubicacion) as longitud,
        cs.fecha_registro, cs.sincronizado, cs.intentos_sincronizacion,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      ${baseQuery}
      ORDER BY cs.fecha_registro DESC
      LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchCortesServicioDetallado:', error);
    throw error;
  }
};

/**
 * Obtiene registros detallados de medidores con filtros y paginación.
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchMedidoresDetallado = async (filters) => {
  try {
    const { usuario_id, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM medidores m JOIN usuarios u ON m.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (usuario_id) {
      baseQuery += ' AND m.usuario_id = ?';
      params.push(usuario_id);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND m.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `
      SELECT
        m.id, m.cuenta, m.medidor, m.lectura, m.foto_fachada_path,
        ST_Y(m.ubicacion) as latitud, ST_X(m.ubicacion) as longitud,
        m.fecha_registro, m.sincronizado,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      ${baseQuery}
      ORDER BY m.fecha_registro DESC
      LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchMedidoresDetallado:', error);
    throw error;
  }
};

/**
 * Obtiene registros detallados de entregas individuales con filtros y paginación.
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchEntregasIndividualesDetallado = async (filters) => {
  try {
    const { usuario_id, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM entregas_individuales ei JOIN usuarios u ON ei.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (usuario_id) {
      baseQuery += ' AND ei.usuario_id = ?';
      params.push(usuario_id);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND ei.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `
      SELECT
        ei.id, ei.cuenta, ei.foto_path, ei.precision_gps,
        ST_Y(ei.ubicacion) as latitud, ST_X(ei.ubicacion) as longitud,
        ei.fecha_registro, ei.sincronizado,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      ${baseQuery}
      ORDER BY ei.fecha_registro DESC
      LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchEntregasIndividualesDetallado:', error);
    throw error;
  }
};

/**
 * Obtiene registros detallados de entregas de edificio con filtros y paginación.
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchEntregasEdificioDetallado = async (filters) => {
  try {
    const { usuario_id, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM entregas_edificio ee JOIN usuarios u ON ee.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (usuario_id) {
      baseQuery += ' AND ee.usuario_id = ?';
      params.push(usuario_id);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND ee.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `
      SELECT
        ee.id, ee.total_cuentas, ee.cuentas_json, ee.foto_path, ee.precision_gps,
        ST_Y(ee.ubicacion) as latitud, ST_X(ee.ubicacion) as longitud,
        ee.fecha_registro, ee.sincronizado,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      ${baseQuery}
      ORDER BY ee.fecha_registro DESC
      LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchEntregasEdificioDetallado:', error);
    throw error;
  }
};

/**
 * Obtiene registros detallados de infraestructura con filtros y paginación.
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchInfraestructuraDetallado = async (filters) => {
  try {
    const { usuario_id, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM infraestructura i JOIN usuarios u ON i.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (usuario_id) {
      baseQuery += ' AND i.usuario_id = ?';
      params.push(usuario_id);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND i.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `
      SELECT
        i.id, i.descripcion, i.foto_factura_path, i.fotos_trabajo_json, i.ruta_audio_path,
        ST_Y(i.ubicacion) as latitud, ST_X(i.ubicacion) as longitud,
        i.fecha_registro, i.sincronizado,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      ${baseQuery}
      ORDER BY i.fecha_registro DESC
      LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchInfraestructuraDetallado:', error);
    throw error;
  }
};

/**
 * Obtiene registros detallados de notificadores con filtros y paginación.
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Object>} Objeto con datos y paginación
 */
export const fetchNotificadoresDetallado = async (filters) => {
  try {
    const { usuario_id, fecha_inicio, fecha_fin } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    validateDateRange(fecha_inicio, fecha_fin);

    let baseQuery = 'FROM notificadores n JOIN usuarios u ON n.usuario_id = u.id WHERE 1=1';
    const params = [];

    if (usuario_id) {
      baseQuery += ' AND n.usuario_id = ?';
      params.push(usuario_id);
    }
    if (fecha_inicio && fecha_fin) {
      baseQuery += ' AND n.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    const countQuery = `SELECT COUNT(*) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0]?.total || 0;

    const dataQuery = `
      SELECT
        n.id, n.observaciones, n.foto_path, n.precision_gps,
        ST_Y(n.ubicacion) as latitud, ST_X(n.ubicacion) as longitud,
        n.fecha_registro, n.sincronizado,
        CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      ${baseQuery}
      ORDER BY n.fecha_registro DESC
      LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
        hasNextPage: pageNum * limitNum < total,
        hasPrevPage: pageNum > 1,
      },
    };
  } catch (error) {
    console.error('Error en fetchNotificadoresDetallado:', error);
    throw error;
  }
};

/**
 * Exporta historial para descarga
 * @param {Object} filters - Filtros de búsqueda
 * @returns {Promise<Array>} Array con datos para exportar
 */
export const exportHistorial = async (filters) => {
  try {
    const { usuario_id, tipo_registro, fecha_inicio, fecha_fin } = filters;

    validateDateRange(fecha_inicio, fecha_fin);

    let query = `
      SELECT vc.*, CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario
      FROM vista_coordenadas vc
      JOIN usuarios u ON vc.usuario_id = u.id
      WHERE 1 = 1
    `;
    const params = [];

    if (usuario_id) {
      query += ' AND vc.usuario_id = ?';
      params.push(usuario_id);
    }
    if (tipo_registro) {
      query += ' AND vc.tipo_registro = ?';
      params.push(tipo_registro);
    }
    if (fecha_inicio && fecha_fin) {
      query += ' AND vc.fecha_registro BETWEEN ? AND ?';
      params.push(fecha_inicio, formatDateForQuery(fecha_fin));
    }

    query += ' ORDER BY vc.fecha_registro DESC';

    const [rows] = await pool.execute(query, params);
    return rows;
  } catch (error) {
    console.error('Error en exportHistorial:', error);
    throw new Error('No se pudo exportar el historial');
  }
};

/**
 * Obtiene todas las asignaciones de trabajo.
 * @returns {Promise<Array>} Lista de asignaciones.
 */
export const fetchAsignaciones = async (filters = {}) => {
  try {
    const { estado, usuario_id, sortBy = 'fecha_asignacion', sortOrder = 'desc' } = filters;
    const { pageNum, limitNum, offset } = getSafePagination(filters);

    const sortableColumns = {
      id: 'a.id',
      tipo_tarea: 'a.titulo', // Corregido para apuntar a la columna correcta
      estado: 'a.estado',
      fecha_limite: 'a.fecha_limite',
      nombre_usuario: 'nombre_usuario',
      fecha_asignacion: 'a.fecha_creacion' // Corregido para usar la columna real
    };
    const sortColumn = sortableColumns[sortBy] || 'a.fecha_asignacion';
    const order = sortOrder.toLowerCase() === 'asc' ? 'ASC' : 'DESC';

    let baseQuery = `
      FROM asignaciones_trabajo a
      JOIN usuarios u ON a.usuario_asignado_id = u.id
      LEFT JOIN usuarios admin ON a.creado_por_id = admin.id
    `;
    const whereClauses = [];
    const params = [];

    if (estado) {
      whereClauses.push('a.estado = ?');
      params.push(estado);
    }
    if (usuario_id) {
      whereClauses.push('a.usuario_asignado_id = ?');
      params.push(usuario_id);
    }

    if (whereClauses.length > 0) {
      baseQuery += ` WHERE ${whereClauses.join(' AND ')}`;
    }

    const countQuery = `SELECT COUNT(a.id) as total ${baseQuery}`;
    const [countResult] = await pool.execute(countQuery, params);
    const total = countResult[0].total;

    const dataQuery = `SELECT a.id, a.titulo, a.tipo_trabajo, a.descripcion, a.estado, a.fecha_creacion as fecha_asignacion, a.fecha_limite, CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario, admin.usuario as admin_nombre ${baseQuery} ORDER BY ${sortColumn} ${order} LIMIT ? OFFSET ?`;
    const dataParams = [...params, limitNum, offset];
    const [rows] = await pool.execute(dataQuery, dataParams);

    return {
      data: rows,
      pagination: {
        total,
        page: pageNum,
        limit: limitNum,
        totalPages: Math.ceil(total / limitNum),
      }
    };
  } catch (error) {
    console.error('Error en fetchAsignaciones:', error);
    throw new Error('No se pudieron obtener las asignaciones');
  }
};

/**
 * Obtiene todas las asignaciones para exportar, aplicando filtros.
 * @param {Object} filters - Filtros de búsqueda.
 * @returns {Promise<Array>} Lista de asignaciones para exportar.
 */
export const exportAsignaciones = async (filters = {}) => {
  try {
    const { estado, usuario_id, sortBy = 'fecha_asignacion', sortOrder = 'desc' } = filters;

    // Whitelist de columnas para ordenar
    const sortableColumns = {
      id: 'a.id',
      tipo_tarea: 'a.titulo', // Corregido para apuntar a la columna correcta
      estado: 'a.estado',
      fecha_limite: 'a.fecha_limite',
      nombre_usuario: 'nombre_usuario',
      fecha_asignacion: 'a.fecha_creacion' // Corregido para usar la columna real
    };
    const sortColumn = sortableColumns[sortBy] || 'a.fecha_asignacion';
    const order = sortOrder.toLowerCase() === 'asc' ? 'ASC' : 'DESC';

    let query = `SELECT a.id, a.titulo, a.tipo_trabajo, a.descripcion, a.estado, a.fecha_creacion as fecha_asignacion, a.fecha_limite, CONCAT(u.nombre, ' ', u.apellido_paterno) as nombre_usuario, admin.usuario as admin_nombre FROM asignaciones_trabajo a JOIN usuarios u ON a.usuario_asignado_id = u.id LEFT JOIN usuarios admin ON a.creado_por_id = admin.id`;
    const whereClauses = [];
    const params = [];

    if (estado) {
      whereClauses.push('a.estado = ?');
      params.push(estado);
    }
    if (usuario_id) {
      whereClauses.push('a.usuario_asignado_id = ?');
      params.push(usuario_id);
    }
    if (whereClauses.length > 0) {
      query += ` WHERE ${whereClauses.join(' AND ')}`;
    }
    query += ` ORDER BY ${sortColumn} ${order}`;
    const [rows] = await pool.execute(query, params);
    return rows;
  } catch (error) {
    console.error('Error en exportAsignaciones:', error);
    throw new Error('No se pudieron obtener las asignaciones para exportar');
  }
};

/**
 * Crea una nueva asignación de trabajo.
 * @param {number} adminId - ID del admin que crea la tarea.
 * @param {Object} asignacionData - Datos de la asignación.
 * @returns {Promise<number>} ID de la nueva asignación.
 */
export const createAsignacion = async (adminId, asignacionData) => {
  const { usuario_id, tipo_trabajo, titulo, descripcion, fecha_limite } = asignacionData;
  const [result] = await pool.execute(
    'INSERT INTO asignaciones_trabajo (usuario_asignado_id, tipo_trabajo, titulo, descripcion, fecha_limite, creado_por_id) VALUES (?, ?, ?, ?, ?, ?)',
    [usuario_id, tipo_trabajo, titulo, descripcion, fecha_limite || null, adminId]
  );
  return result.insertId;
};

/**
 * Actualiza una asignación de trabajo existente.
 * @param {number} adminId - ID del admin que actualiza.
 * @param {number} asignacionId - ID de la asignación a actualizar.
 * @param {Object} asignacionData - Nuevos datos para la asignación.
 * @returns {Promise<Object>} Resultado de la operación de actualización.
 */
export const updateAsignacion = async (adminId, asignacionId, asignacionData) => {
  const { usuario_id, tipo_trabajo, titulo, descripcion, fecha_limite } = asignacionData;
  const [result] = await pool.execute(
    'UPDATE asignaciones_trabajo SET usuario_asignado_id = ?, tipo_trabajo = ?, titulo = ?, descripcion = ?, fecha_limite = ? WHERE id = ?',
    [usuario_id, tipo_trabajo, titulo, descripcion, fecha_limite || null, asignacionId]
  );
  return result;
};

/**
 * Actualiza solo el estado de una asignación.
 * @param {number} adminId - ID del admin que actualiza.
 * @param {number} asignacionId - ID de la asignación.
 * @param {string} estado - Nuevo estado.
 * @returns {Promise<Object>} Resultado de la operación.
 */
export const updateEstadoAsignacion = async (adminId, asignacionId, estado) => {
  const [result] = await pool.execute(
    'UPDATE asignaciones_trabajo SET estado = ? WHERE id = ?',
    [estado, asignacionId]
  );
  return result;
};

/**
 * Elimina un usuario de la base de datos.
 * @param {number} adminId - ID del admin que realiza la acción.
 * @param {number} userId - ID del usuario a eliminar.
 * @returns {Promise<Object>} Resultado de la operación.
 */
export const deleteUser = async (adminId, userId) => {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    // Opcional: Verificar si el usuario a eliminar no es el mismo que el admin.
    if (Number(adminId) === Number(userId)) {
      throw new Error('Un administrador no se puede eliminar a sí mismo.');
    }

    const [result] = await conn.execute('DELETE FROM usuarios WHERE id = ?', [userId]);

    if (result.affectedRows > 0) {
      await logAdminAction(adminId, ADMIN_ACTIONS.DELETE_RECORD, {
        recurso_afectado: 'usuarios',
        recurso_id: userId,
      });
    }

    await conn.commit();
    return result;
  } catch (error) {
    await conn.rollback();
    console.error('Error en deleteUser:', error);
    // Lanza el error para que el controlador lo capture.
    throw new Error('No se pudo eliminar el usuario. Es posible que tenga registros asociados.');
  } finally {
    conn.release();
  }
};


/**
 * Obtiene las estadísticas de productividad para el dashboard.
 * @returns {Promise<Array>}
 */
export const fetchProductividadDashboard = async () => {
  try {
    const [rows] = await pool.execute('SELECT * FROM vista_productividad_dashboard ORDER BY total_actividades DESC');
    return rows;
  } catch (error) {
    console.error('Error en fetchProductividadDashboard:', error);
    throw new Error('No se pudo obtener la productividad del dashboard');
  }
};

/**
 * Obtiene las estadísticas mensuales.
 * @returns {Promise<Array>}
 */
export const fetchEstadisticasMensuales = async () => {
  try {
    const [rows] = await pool.execute('SELECT * FROM vista_estadisticas_mensuales');
    return rows;
  } catch (error) {
    console.error('Error en fetchEstadisticasMensuales:', error);
    throw new Error('No se pudieron obtener las estadísticas mensuales');
  }
};

/**
 * Obtiene el rendimiento general.
 * @returns {Promise<Object>}
 */
export const fetchRendimientoGeneral = async () => {
  try {
    const [rows] = await pool.execute('SELECT * FROM vista_rendimiento_general');
    return rows[0] || {};
  } catch (error) {
    console.error('Error en fetchRendimientoGeneral:', error);
    throw new Error('No se pudo obtener el rendimiento general');
  }
};

/**
 * Obtiene el porcentaje de sincronización por tipo de registro.
 * @returns {Promise<Array>}
 */
export const fetchPorcentajeSincronizacion = async () => {
  try {
    const [rows] = await pool.execute('SELECT * FROM vista_porcentaje_sincronizacion');
    return rows;
  } catch (error) {
    console.error('Error en fetchPorcentajeSincronizacion:', error);
    throw new Error('No se pudo obtener el porcentaje de sincronización');
  }
};

/**
 * Obtiene las alertas operativas de sincronización.
 * @returns {Promise<Array>}
 */
export const fetchAlertasOperativas = async () => {
  try {
    const [rows] = await pool.execute('SELECT * FROM vista_alertas_operativas');
    return rows;
  } catch (error) {
    console.error('Error en fetchAlertasOperativas:', error);
    throw new Error('No se pudieron obtener las alertas operativas');
  }
};

/**
 * Obtiene los usuarios más y menos productivos.
 * @returns {Promise<Object>}
 */
export const fetchRankingProductividadCompleto = async () => {
  try {
    const [masProductivos] = await pool.execute('SELECT * FROM vista_usuarios_mas_productivos');
    const [menosProductivos] = await pool.execute('SELECT * FROM vista_usuarios_menos_productivos');
    return {
      mas_productivos: masProductivos,
      menos_productivos: menosProductivos
    };
  } catch (error) {
    console.error('Error en fetchRankingProductividadCompleto:', error);
    throw new Error('No se pudo obtener el ranking de productividad');
  }
};

/**
 * Obtiene datos de una vista de la base de datos de forma segura.
 * @param {string} viewName - El nombre de la vista a consultar.
 * @returns {Promise<Array>} - Una promesa que resuelve a un array de resultados.
 */
export const fetchFromView = async (viewName) => {
  // La validación del nombre de la vista ya se hace en el controlador (lista blanca).
  // Esto previene inyección SQL.
  const [rows] = await pool.query(`SELECT * FROM ${viewName}`);
  return rows;
};