import { Router } from 'express';
import {
  getResumenActividades,
  getEstadisticasDiarias,
  getCoordenadas,
  getHistorialUsuario,
  getRegistrosCercanos,
  getHistorialUbicacionesPorFecha, // 1. Importa la nueva función
} from '../controllers/reportes.controller.js';
import { validate } from '../middlewares/validator.middleware.js';
import { historialUsuarioSchema, coordenadasSchema, historialUbicacionesSchema } from '../schemas/reportes.schema.js';

const router = Router();

/**
 * @swagger
 * tags:
 *   name: Reportes
 *   description: Endpoints para obtener reportes y estadísticas del usuario autenticado.
 */

/**
 * @swagger
 * /reportes/resumen-actividades:
 *   get:
 *     summary: Obtiene un resumen de las actividades del usuario autenticado.
 *     tags: [Reportes]
 *     responses:
 *       200:
 *         description: Un objeto con el total de actividades del usuario.
 */
router.get('/resumen-actividades', getResumenActividades);

/**
 * @swagger
 * /reportes/estadisticas-diarias:
 *   get:
 *     summary: Obtiene estadísticas diarias de actividad para el usuario autenticado.
 *     tags: [Reportes]
 *     responses:
 *       200:
 *         description: Una lista de actividades agrupadas por día y tipo.
 */
router.get('/estadisticas-diarias', getEstadisticasDiarias);

/**
 * @swagger
 * /reportes/coordenadas:
 *   get:
 *     summary: Obtiene coordenadas para un mapa de calor, con filtros opcionales.
 *     tags: [Reportes]
 *     parameters:
 *       - in: query
 *         name: tipo_registro
 *         schema:
 *           type: string
 *         description: Tipo de registro para filtrar.
 *       - in: query
 *         name: fecha_inicio
 *         schema:
 *           type: string
 *           format: date
 *         description: Fecha de inicio del rango (YYYY-MM-DD).
 *       - in: query
 *         name: fecha_fin
 *         schema:
 *           type: string
 *           format: date
 *         description: Fecha de fin del rango (YYYY-MM-DD).
 *     responses:
 *       200:
 *         description: Una lista de objetos con latitud y longitud.
 */
router.get('/coordenadas', validate(coordenadasSchema, 'query'), getCoordenadas);

/**
 * @swagger
 * /reportes/historial:
 *   get:
 *     summary: Obtiene el historial de actividades del usuario autenticado, con paginación.
 *     tags: [Reportes]
 *     parameters:
 *       - in: query
 *         name: tipo_registro
 *         schema:
 *           type: string
 *         description: Tipo de registro para filtrar.
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 20
 *     responses:
 *       200:
 *         description: Una lista paginada de las actividades del usuario.
 */
router.get('/historial', validate(historialUsuarioSchema, 'query'), getHistorialUsuario);

router.get('/registros-cercanos', getRegistrosCercanos);

router.get('/historial-ubicaciones', validate(historialUbicacionesSchema, 'query'), getHistorialUbicacionesPorFecha);

export default router;