import { Router } from 'express';
import {
  getResumenGlobal,
  getRankingProductividad,
  getActividadReciente,
  getHistorialGlobalFiltrado,
  getEstadoSincronizacion,
  getUltimosRegistrosPorUsuario,
  getUsuarios,
  getUsuarioPerfil,
  crearUsuario,
  updateEstadoUsuario,
  updateUsuario,
  deleteRegistro,
  getConfiguraciones,
  updateConfiguraciones,
  getActividadPorHora,
  resetPasswordUsuario,
  getRegistrosPorTipo,
  exportarRecurso,
  getAuditLog,
  generarPasswordTemporal,
  getAsignaciones,
  crearAsignacion,
  actualizarAsignacion,
  actualizarEstadoAsignacion,
  getRutasUsuarios,
  getTopUsuariosPorAsignaciones,
  getActividadPorDiaSemana,
  getUbicacionesEnVivo,
  getCortesServicioDetallado,
  getMedidoresDetallado,
  getEntregasIndividualesDetallado,
  getEntregasEdificioDetallado,
  getInfraestructuraDetallado,
  getNotificadoresDetallado,
  getProductividadDashboard,
  getEstadisticasMensuales,
  getRendimientoGeneral,
  getPorcentajeSincronizacion,
  getRankingProductividadCompleto,
  deleteUsuario
} from '../controllers/admin.controller.js';
import { validate } from '../middlewares/validator.middleware.js';
import { historialGlobalSchema, usuarioIdSchema, updateEstadoUsuarioSchema, updateUsuarioSchema, crearUsuarioSchema, deleteRegistroSchema, updateConfiguracionesSchema, analiticasSchema, resetPasswordSchema, crearAsignacionSchema, updateAsignacionSchema, updateEstadoAsignacionSchema, asignacionesFiltroSchema, auditLogSchema, rutasUsuariosSchema } from '../schemas/admin.schema.js';

const router = Router();

/**
 * @swagger
 * tags:
 *   name: Admin
 *   description: Endpoints para el dashboard de administración.
 */

/**
 * @swagger
 * /admin/resumen-global:
 *   get:
 *     summary: Obtiene un resumen de los KPIs globales del sistema.
 *     tags: [Admin]
 *     responses:
 *       200:
 *         description: Un objeto con los KPIs principales.
 */
router.get('/resumen-global', getResumenGlobal);

/**
 * @swagger
 * /admin/ranking-productividad:
 *   get:
 *     summary: Obtiene un ranking de productividad de todos los usuarios.
 *     tags: [Admin]
 *     responses:
 *       200:
 *         description: Una lista de usuarios ordenada por su total de actividades.
 */
router.get('/ranking-productividad', getRankingProductividad);

router.get('/ubicaciones-en-vivo', getUbicacionesEnVivo);

/**
 * @swagger
 * /admin/actividad-reciente:
 *   get:
 *     summary: Obtiene un feed con las últimas 15 actividades registradas en el sistema.
 *     tags: [Admin]
 *     responses:
 *       200:
 *         description: Una lista de las actividades más recientes.
 */
router.get('/actividad-reciente', getActividadReciente);

/**
 * @swagger
 * /admin/historial-filtrado:
 *   get:
 *     summary: Obtiene un historial global de todos los registros, con filtros y paginación.
 *     tags: [Admin]
 *     parameters:
 *       - in: query
 *         name: usuario_id
 *         schema:
 *           type: integer
 *         description: ID del usuario para filtrar los registros.
 *       - in: query
 *         name: tipo_registro
 *         schema:
 *           type: string
 *         description: Tipo de registro para filtrar (ej. 'medidor', 'corte_servicio').
 *       - in: query
 *         name: fecha_inicio
 *         schema:
 *           type: string
 *           format: date
 *         description: Fecha de inicio del rango de búsqueda (YYYY-MM-DD).
 *       - in: query
 *         name: fecha_fin
 *         schema:
 *           type: string
 *           format: date
 *         description: Fecha de fin del rango de búsqueda (YYYY-MM-DD).
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Número de página para la paginación.
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 20
 *         description: Cantidad de resultados por página.
 *     responses:
 *       200:
 *         description: Una lista paginada de registros del historial.
 */
router.get('/historial-filtrado', validate(historialGlobalSchema, 'query'), getHistorialGlobalFiltrado);

/**
 * @swagger
 * /admin/estado-sincronizacion:
 *   get:
 *     summary: Obtiene el estado de sincronización de cada usuario activo.
 *     tags: [Admin]
 *     responses:
 *       200:
 *         description: Una lista de usuarios con la cantidad de registros pendientes por sincronizar.
 */
router.get('/estado-sincronizacion', getEstadoSincronizacion);

/**
 * @swagger
 * /admin/ultimo-registro-usuario/{usuario_id}:
 *   get:
 *     summary: Obtiene el último registro de cada tipo para un usuario específico.
 *     tags: [Admin]
 *     parameters:
 *       - in: path
 *         name: usuario_id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID del usuario a consultar.
 *     responses:
 *       200:
 *         description: Un objeto con el último registro de cada tipo de actividad.
 *       404:
 *         description: Usuario no encontrado.
 */
router.get('/ultimo-registro-usuario/:id', validate(usuarioIdSchema, 'params'), getUltimosRegistrosPorUsuario);

/**
 * @swagger
 * /admin/usuarios:
 *   get:
 *     summary: Obtiene una lista de todos los usuarios del sistema.
 *     tags: [Admin]
 *     responses:
 *       200:
 *         description: Una lista completa de usuarios.
 */
router.get('/usuarios', getUsuarios);

router.get('/usuarios/:id/perfil', validate(usuarioIdSchema, 'params'), getUsuarioPerfil);

router.post('/usuarios', validate(crearUsuarioSchema), crearUsuario);

/**
 * @swagger
 * /admin/usuarios/{usuario_id}/estado:
 *   put:
 *     summary: Actualiza el estado de un usuario (activo/inactivo).
 *     tags: [Admin]
 *     parameters:
 *       - in: path
 *         name: usuario_id
 *         required: true
 *         schema:
 *           type: integer
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               estatus_id:
 *                 type: integer
 *                 description: 0 para inactivo, 1 para activo.
 *     responses:
 *       200:
 *         description: Estado del usuario actualizado.
 *       404:
 *         description: Usuario no encontrado.
 */
router.put('/usuarios/:id/estado', validate(usuarioIdSchema, 'params'), updateEstadoUsuario);

router.put('/usuarios/:id', validate(usuarioIdSchema, 'params'), validate(updateUsuarioSchema), updateUsuario);

router.delete('/usuarios/:id', validate(usuarioIdSchema, 'params'), deleteUsuario);

router.post('/usuarios/:id/generar-password', validate(usuarioIdSchema, 'params'), generarPasswordTemporal);

router.put('/usuarios/:id/reset-password', validate(usuarioIdSchema, 'params'), validate(resetPasswordSchema), resetPasswordUsuario);

/**
 * @swagger
 * /admin/registros/{tabla}/{id}:
 *   delete:
 *     summary: Elimina un registro específico de una tabla.
 *     tags: [Admin]
 *     parameters:
 *       - in: path
 *         name: tabla
 *         required: true
 *         schema:
 *           type: string
 *         description: Nombre de la tabla (ej. 'cortes_servicio').
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID del registro a eliminar.
 *     responses:
 *       200:
 *         description: Registro eliminado exitosamente.
 *       404:
 *         description: Registro no encontrado.
 */
router.delete('/registros/:tabla/:id', validate(deleteRegistroSchema, 'params'), deleteRegistro);

router.get('/configuraciones', getConfiguraciones);

router.put('/configuraciones', validate(updateConfiguracionesSchema), updateConfiguraciones);

router.get('/analiticas/actividad-por-hora', validate(analiticasSchema, 'query'), getActividadPorHora);

router.get('/analiticas/registros-por-tipo', validate(analiticasSchema, 'query'), getRegistrosPorTipo);

router.get('/analiticas/rutas-usuarios', validate(rutasUsuariosSchema, 'query'), getRutasUsuarios);

router.get('/analiticas/top-usuarios-asignaciones', validate(analiticasSchema, 'query'), getTopUsuariosPorAsignaciones);

router.get('/analiticas/actividad-por-dia', validate(analiticasSchema, 'query'), getActividadPorDiaSemana);

/**
 * @swagger
 * /admin/exportar/{recurso}:
 *   get:
 *     summary: Exporta datos de un recurso a formato CSV o XLSX.
 *     tags: [Admin]
 *     parameters:
 *       - in: path
 *         name: recurso
 *         required: true
 *         schema:
 *           type: string,
 *           enum: [historial, ranking-productividad, usuarios, asignaciones]
 *         description: El recurso a exportar.
 *       - in: query
 *         name: formato
 *         schema:
 *           type: string
 *           enum: [csv, xlsx]
 *           default: csv
 *         description: El formato del archivo a exportar.
 *     responses:
 *       200:
 *         description: El archivo solicitado para descargar.
 *       404:
 *         description: No se encontraron datos para exportar.
 */
router.get('/exportar/:recurso', validate(historialGlobalSchema, 'query'), exportarRecurso);

router.get('/auditoria', validate(auditLogSchema, 'query'), getAuditLog);

// Rutas para Asignaciones
router.get('/asignaciones', validate(asignacionesFiltroSchema, 'query'), getAsignaciones);
router.post('/asignaciones', validate(crearAsignacionSchema), crearAsignacion);
router.put('/asignaciones/:id', validate(updateAsignacionSchema), actualizarAsignacion);
router.put('/asignaciones/:id/estado', validate(updateEstadoAsignacionSchema), actualizarEstadoAsignacion);

// Nuevas rutas para obtener detalles de registros específicos
router.get('/cortes-servicio-detallado', validate(historialGlobalSchema, 'query'), getCortesServicioDetallado);
router.get('/medidores-detallado', validate(historialGlobalSchema, 'query'), getMedidoresDetallado);
router.get('/entregas-individuales-detallado', validate(historialGlobalSchema, 'query'), getEntregasIndividualesDetallado);
router.get('/entregas-edificio-detallado', validate(historialGlobalSchema, 'query'), getEntregasEdificioDetallado);
router.get('/infraestructura-detallado', validate(historialGlobalSchema, 'query'), getInfraestructuraDetallado);
router.get('/notificadores-detallado', validate(historialGlobalSchema, 'query'), getNotificadoresDetallado);
// Rutas para las nuevas vistas del dashboard
router.get('/dashboard/productividad', getProductividadDashboard);
router.get('/dashboard/estadisticas-mensuales', getEstadisticasMensuales);
router.get('/dashboard/rendimiento-general', getRendimientoGeneral);
router.get('/dashboard/porcentaje-sincronizacion', getPorcentajeSincronizacion);
router.get('/dashboard/ranking-productividad-completo', getRankingProductividadCompleto);


export default router;
