// Importa la librería `multer` para manejar la subida de archivos (multipart/form-data).
import multer from 'multer';
// Importa el módulo `path` de Node.js para trabajar con rutas de archivos.
import path from 'path';

// --- CONFIGURACIÓN DE ALMACENAMIENTO DE MULTER ---
// Se usa `memoryStorage` para guardar el archivo subido como un Buffer en la memoria RAM.
// Esto es ideal porque nos permite procesar el archivo (ej. comprimir una imagen)
// antes de decidir dónde y cómo guardarlo en el disco.
const storage = multer.memoryStorage();

// --- CREACIÓN DEL MIDDLEWARE DE UPLOAD ---
export const upload = multer({
  storage, // Usa la configuración de almacenamiento en memoria.
  limits: { fileSize: 10 * 1024 * 1024 }, // Límite de tamaño de archivo: 10 MB.

  // `fileFilter` es una función para aceptar o rechazar un archivo.
  fileFilter: (req, file, cb) => {
    // Define una lista blanca de extensiones de archivo permitidas.
    const allowed = ['.png', '.jpg', '.jpeg', '.gif', '.webp', '.mp3', '.wav', '.m4a', '.aac', '.ogg'];
    // Obtiene la extensión del archivo original y la convierte a minúsculas.
    const ext = path.extname(file.originalname).toLowerCase();
    // Si la extensión no está en la lista blanca, rechaza el archivo con un error.
    if (!allowed.includes(ext)) return cb(new Error('Tipo de archivo no permitido'), false);
    // Si la extensión es válida, acepta el archivo.
    cb(null, true);
  }
});