import * as AdminService from '../services/admin.service.js';
import { sendExportFile } from '../utils/export.util.js';
import { USER_STATUS } from '../config/constants.js';
import { userConnections } from '../websockets.js'; // Importar el mapa de conexiones compartido
import crypto from 'crypto';


// --- OBTENER RESUMEN GLOBAL PARA EL DASHBOARD ---
export const getResumenGlobal = async (req, res) => {
  const resumen = await AdminService.fetchResumenGlobal();
  res.json(resumen);
};

// --- OBTENER RANKING DE PRODUCTIVIDAD DE USUARIOS ---
export const getRankingProductividad = async (req, res) => {
  const rows = await AdminService.fetchRankingProductividad();
  res.json(rows);
};

// --- OBTENER FEED DE ACTIVIDAD RECIENTE ---
export const getActividadReciente = async (req, res) => {
  const rows = await AdminService.fetchActividadReciente();
  res.json(rows);
};

// --- OBTENER UBICACIONES EN VIVO DE USUARIOS ---
export const getUbicacionesEnVivo = async (req, res) => {
  const data = await AdminService.fetchUbicacionesEnVivo();
  res.json(data);
};

// --- OBTENER HISTORIAL GLOBAL FILTRADO Y PAGINADO ---
export const getHistorialGlobalFiltrado = async (req, res) => {
  const resultado = await AdminService.fetchHistorialGlobalFiltrado(req.query);
  res.json(resultado);
};

// --- OBTENER ESTADO DE SINCRONIZACIÓN POR USUARIO ---
export const getEstadoSincronizacion = async (req, res) => {
  const rows = await AdminService.fetchEstadoSincronizacion();
  res.json(rows);
};

// --- OBTENER ÚLTIMO REGISTRO DE CADA TIPO PARA UN USUARIO ESPECÍFICO ---
export const getUltimosRegistrosPorUsuario = async (req, res) => {
  const { id } = req.params;
  const ultimosRegistros = await AdminService.fetchUltimosRegistrosPorUsuario(id);
  res.json(ultimosRegistros);
};

// --- OBTENER LISTA DE TODOS LOS USUARIOS ---
export const getUsuarios = async (req, res) => {
  const result = await AdminService.fetchUsuarios(req.query);
  res.json(result);
};



// --- OBTENER PERFIL COMPLETO DE UN USUARIO ---
export const getUsuarioPerfil = async (req, res) => {
  // Asegurarse de que el ID sea un número entero antes de pasarlo al servicio.
  const userId = parseInt(req.params.id, 10);
  const perfil = await AdminService.fetchUserProfile(userId);
  res.json(perfil);
};

// --- ACTUALIZAR EL ESTADO DE UN USUARIO (ACTIVAR/DESACTIVAR) ---
export const updateEstadoUsuario = async (req, res) => {
  const { id } = req.params;
  const { estatus_id } = req.body;

  // Validación manual para asegurar que estatus_id es 0 o 1.
  if (estatus_id === undefined || (estatus_id !== 0 && estatus_id !== 1)) {
    return res.status(400).json({ error: 'El campo "estatus_id" es requerido y debe ser 0 o 1.' });
  }

  const adminId = req.user.id;
  const userId = parseInt(id, 10); // Convertimos el id de la ruta a número
  const result = await AdminService.updateUserStatus(adminId, userId, estatus_id);
  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Usuario no encontrado.' });
  }
  res.json({ mensaje: 'Estado del usuario actualizado correctamente.' });
};

// --- ACTUALIZAR DETALLES DE UN USUARIO ---
export const updateUsuario = async (req, res) => {
  const userId = parseInt(req.params.id, 10);
  const adminId = req.user.id;

  const result = await AdminService.updateUserDetails(adminId, userId, req.body);
  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Usuario no encontrado.' });
  }
  res.json({ mensaje: 'Datos del usuario actualizados correctamente.' });
};

// --- CREAR UN NUEVO USUARIO (POR UN ADMIN) ---
export const crearUsuario = async (req, res) => {
  const adminId = req.user.id;
  try {
    const nuevoUsuario = await AdminService.createNewUser(adminId, req.body);
    res.status(201).json(nuevoUsuario);
  } catch (error) {
    // Captura el error específico de 'nombre de usuario ya existe'
    if (error.message === 'El nombre de usuario ya existe') {
      return res.status(409).json({ error: error.message });
    } else if (error.code === 'ER_DUP_ENTRY') {
      return res.status(409).json({ error: 'El usuario o número de empleado ya existe.' });
    }
    throw error; // Dejar que el errorHandler general lo capture
  }
};

// --- OBTENER CONFIGURACIONES DEL SISTEMA ---
export const getConfiguraciones = async (req, res) => {
  const configs = await AdminService.fetchConfiguraciones();
  res.json(configs);
};

// --- ELIMINAR UN REGISTRO ESPECÍFICO ---
export const deleteRegistro = async (req, res) => {
  const { tabla, id } = req.params;
  const adminId = req.user.id;
  const result = await AdminService.deleteRecord(adminId, tabla, id);
  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Registro no encontrado o ya fue eliminado.' });
  }
  res.json({ mensaje: `Registro con ID ${id} de la tabla ${tabla} ha sido eliminado.` });
};

// --- ACTUALIZAR CONFIGURACIONES DEL SISTEMA ---
export const updateConfiguraciones = async (req, res) => {
  const adminId = req.user.id;
  await AdminService.updateConfiguraciones(adminId, req.body);
  res.json({ mensaje: 'Configuraciones actualizadas correctamente.' });
};

// --- OBTENER ANALÍTICAS DE ACTIVIDAD POR HORA ---
export const getActividadPorHora = async (req, res) => {
  const data = await AdminService.fetchActividadPorHora(req.query);
  res.json(data);
};

// --- OBTENER RUTAS DE USUARIOS POR FECHA ---
export const getRutasUsuarios = async (req, res) => {
  const data = await AdminService.fetchRutasUsuarios(req.query);
  res.json(data);
};

// --- OBTENER TOP USUARIOS POR ASIGNACIONES ---
export const getTopUsuariosPorAsignaciones = async (req, res) => {
  const data = await AdminService.fetchTopUsuariosPorAsignaciones();
  res.json(data);
};

// --- OBTENER ACTIVIDAD POR DÍA DE LA SEMANA ---
export const getActividadPorDiaSemana = async (req, res) => {
  const data = await AdminService.fetchActividadPorDiaSemana(req.query);
  res.json(data);
};

// --- RESTABLECER CONTRASEÑA DE UN USUARIO ---
export const resetPasswordUsuario = async (req, res) => {
  const { id } = req.params;
  const { password } = req.body;
  const adminId = req.user.id;

  const result = await AdminService.resetUserPassword(adminId, id, password);

  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Usuario no encontrado.' });
  }

  res.json({ mensaje: 'Contraseña del usuario actualizada correctamente.' });
};

// --- GENERAR Y RESTABLECER CONTRASEÑA TEMPORAL PARA UN USUARIO ---
export const generarPasswordTemporal = async (req, res) => {
  const { id } = req.params;
  const adminId = req.user.id;

  // 1. Generar una contraseña temporal segura
  const tempPassword = crypto.randomBytes(8).toString('hex');

  // 2. Actualizar la contraseña del usuario en la base de datos
  const result = await AdminService.resetUserPassword(adminId, id, tempPassword);

  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Usuario no encontrado.' });
  }

  // 3. Devolver la contraseña temporal para que el frontend la muestre
  res.json({
    message: `La contraseña para el usuario ha sido restablecida.`,
    tempPassword: tempPassword,
  });
};

// --- OBTENER ANALÍTICAS DE REGISTROS POR TIPO ---
export const getRegistrosPorTipo = async (req, res) => {
  const data = await AdminService.fetchRegistrosPorTipo(req.query);
  res.json(data);
};

// --- EXPORTAR RECURSOS A CSV/EXCEL ---
export const exportarRecurso = async (req, res) => {
  const { recurso } = req.params;
  const { formato = 'csv' } = req.query;

  let data;
  let columns;

  // Define qué datos y columnas usar según el recurso solicitado.
  switch (recurso) {
    case 'historial':
      data = await AdminService.exportHistorial(req.query);
      columns = [
        { header: 'ID', key: 'id', width: 10 },
        { header: 'Tipo Registro', key: 'tipo_registro', width: 20 },
        { header: 'Referencia', key: 'referencia', width: 20 },
        { header: 'Usuario', key: 'nombre_usuario', width: 30 },
        { header: 'Fecha Registro', key: 'fecha_registro', width: 25 },
        { header: 'Latitud', key: 'latitud', width: 15 },
        { header: 'Longitud', key: 'longitud', width: 15 },
      ];
      break;
    case 'ranking-productividad':
      data = await AdminService.fetchRankingProductividad();
      columns = [
        { header: 'ID Usuario', key: 'usuario_id', width: 15 },
        { header: 'Nombre Completo', key: 'nombre_completo', width: 30 },
        { header: 'Total Actividades', key: 'total_actividades', width: 20 },
        { header: 'Total Cortes', key: 'total_cortes', width: 15 },
        { header: 'Total Lecturas', key: 'total_medidores', width: 15 },
        { header: 'Total Notificaciones', key: 'total_notificadores', width: 20 },
      ];
      break;
    case 'usuarios':
      data = await AdminService.fetchUsuarios();
      columns = [
        { header: 'ID', key: 'id', width: 10 },
        { header: 'Nombre', key: 'nombre', width: 20 },
        { header: 'Apellido Paterno', key: 'apellido_paterno', width: 20 },
        { header: 'Usuario', key: 'usuario', width: 20 },
        { header: 'Numero Empleado', key: 'numero_empleado', width: 20 },
        { header: 'Estado ID', key: 'estatus_id', width: 15 },
        { header: 'Ultimo Login', key: 'ultimo_login', width: 25 },
      ];
      break;
    case 'asignaciones':
      data = await AdminService.exportAsignaciones(req.query);
      columns = [
        { header: 'ID', key: 'id', width: 10 },
        { header: 'Usuario Asignado', key: 'nombre_usuario', width: 30 },
        { header: 'Tipo Tarea', key: 'tipo_tarea', width: 25 },
        { header: 'Referencia', key: 'referencia_tarea', width: 20 },
        { header: 'Estado', key: 'estado', width: 15 },
        { header: 'Fecha Asignación', key: 'fecha_asignacion', width: 25 },
        { header: 'Fecha Límite', key: 'fecha_limite', width: 20 },
      ];
      break;
    default:
      return res.status(400).json({ error: 'El recurso solicitado para exportar no es válido.' });
  }

  if (data.length === 0) {
    return res.status(404).json({ error: 'No hay datos para exportar con los filtros seleccionados.' });
  }

  // Llama a la utilidad para generar y enviar el archivo.
  await sendExportFile(res, data, formato, recurso, columns);
};

// --- OBTENER LOG DE AUDITORÍA ---
export const getAuditLog = async (req, res) => {
  const log = await AdminService.fetchAuditLog(req.query);
  res.json(log);
};

// --- OBTENER ASIGNACIONES DE TRABAJO ---
export const getAsignaciones = async (req, res) => {
  const asignaciones = await AdminService.fetchAsignaciones(req.query);
  res.json(asignaciones);
};

// --- CREAR UNA NUEVA ASIGNACIÓN DE TRABAJO ---
export const crearAsignacion = async (req, res) => {
  const adminId = req.user.id;
  const nuevaAsignacionId = await AdminService.createAsignacion(adminId, req.body);
  res.status(201).json({ id: nuevaAsignacionId, mensaje: 'Asignación creada exitosamente.' });
};

// --- ACTUALIZAR UNA ASIGNACIÓN DE TRABAJO ---
export const actualizarAsignacion = async (req, res) => {
  const { id } = req.params;
  const adminId = req.user.id;
  const result = await AdminService.updateAsignacion(adminId, id, req.body);
  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Asignación no encontrada.' });
  }
  res.json({ mensaje: 'Asignación actualizada correctamente.' });
};

// --- ACTUALIZAR ESTADO DE UNA ASIGNACIÓN ---
export const actualizarEstadoAsignacion = async (req, res) => {
  const { id } = req.params;
  const { estado } = req.body;
  const adminId = req.user.id;
  const result = await AdminService.updateEstadoAsignacion(adminId, id, estado);
  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Asignación no encontrada.' });
  }
  res.json({ mensaje: `Estado de la asignación actualizado a "${estado}".` });
};

// --- FORZAR SINCRONIZACIÓN DE UN USUARIO ---
export const forzarSincronizacion = (req, res) => {
  const { id } = req.params;
  const targetUserId = parseInt(id, 10);

  // Buscar la conexión WebSocket del usuario objetivo
  const ws = userConnections.get(targetUserId);

  if (ws && ws.readyState === ws.OPEN) {
    // Si el usuario está conectado, enviarle el comando
    ws.send(JSON.stringify({ type: 'force_sync' }));
    res.json({ mensaje: `Comando de sincronización forzada enviado al usuario ${targetUserId}.` });
  } else {
    // Si no está conectado, informar al administrador
    res.status(404).json({ error: `El usuario ${targetUserId} no está conectado actualmente.` });
  }
};

export const getProductividadDashboard = async (req, res) => {
  const data = await AdminService.fetchProductividadDashboard();
  res.json(data);
};

export const getEstadisticasMensuales = async (req, res) => {
  const data = await AdminService.fetchEstadisticasMensuales();
  res.json(data);
};

export const getRendimientoGeneral = async (req, res) => {
  const data = await AdminService.fetchRendimientoGeneral();
  res.json(data);
};

export const getPorcentajeSincronizacion = async (req, res) => {
  const data = await AdminService.fetchPorcentajeSincronizacion();
  res.json(data);
};

export const getAlertasOperativas = async (req, res) => {
  const data = await AdminService.fetchAlertasOperativas();
  res.json(data);
};

export const getRankingProductividadCompleto = async (req, res) => {
  const data = await AdminService.fetchRankingProductividadCompleto();
  res.json(data);
};

export const getCortesServicioDetallado = async (req, res) => {
  const data = await AdminService.fetchCortesServicioDetallado(req.query);
  res.json(data);
};

export const getMedidoresDetallado = async (req, res) => {
  const data = await AdminService.fetchMedidoresDetallado(req.query);
  res.json(data);
};

export const getEntregasIndividualesDetallado = async (req, res) => {
  const data = await AdminService.fetchEntregasIndividualesDetallado(req.query);
  res.json(data);
};

export const getEntregasEdificioDetallado = async (req, res) => {
  const data = await AdminService.fetchEntregasEdificioDetallado(req.query);
  res.json(data);
};

export const getInfraestructuraDetallado = async (req, res) => {
  const data = await AdminService.fetchInfraestructuraDetallado(req.query);
  res.json(data);
};

export const getNotificadoresDetallado = async (req, res) => {
  const data = await AdminService.fetchNotificadoresDetallado(req.query);
  res.json(data);
};

export const deleteUsuario = async (req, res) => {
  const { id } = req.params;
  const adminId = req.user.id;
  const result = await AdminService.deleteUser(adminId, id);
  if (result.affectedRows === 0) {
    return res.status(404).json({ error: 'Usuario no encontrado.' });
  }
  res.json({ mensaje: 'Usuario eliminado correctamente.' });
};

// --- CONTROLADOR GENÉRICO PARA VISTAS DEL DASHBOARD ---
export const getGenericViewData = async (req, res) => {
  const { viewName } = req.params;

  // Lista blanca de vistas permitidas para mayor seguridad
  const allowedViews = [
    'vista_usuarios_mas_productivos',
    'vista_usuarios_menos_productivos',
    // Puedes añadir más vistas seguras aquí en el futuro
  ];

  if (!allowedViews.includes(viewName)) {
    return res.status(403).json({ error: 'Acceso a esta vista no está permitido.' });
  }

  // Llama a un servicio genérico que obtiene los datos de la vista
  // (Asegúrate de que este método exista en tu admin.service.js)
  const data = await AdminService.fetchFromView(viewName, req.query);
  res.json({ data });
};